package config

import (
	"encoding/json"
	"fmt"
	"os"
	"path/filepath"
	"strings"

	"github.com/spf13/viper"
	"github.com/zeromicro/go-zero/core/logx"
)

type Config struct {
	DataSource string     `mapstructure:"data_source"`
	Chainmaker Chainmaker `mapstructure:"chainmaker"`
	CNBNDID    string     `mapstructure:"cnbn-did"`
	ApiManage  string     `mapstructure:"api-manage"`
	Type       string     `mapstructure:"type"`
	Log        Log        `mapstructure:"log"`
}

type TemplateConfig struct {
	TemplateId        string                 `mapstructure:"templateId"`
	CredentialSubject map[string]interface{} `mapstructure:"credentialSubject"`
}

type Chainmaker struct {
	SdkConfigPath string `mapstructure:"sdk_config_path"`
	ContractName  string `mapstructure:"contract_name"`
}

type Log struct {
	ServiceName         string `mapstructure:"service_name"`
	Mode                string `mapstructure:"mode"`
	Encoding            string `mapstructure:"encoding"`
	TimeFormat          string `mapstructure:"time_format"`
	Path                string `mapstructure:"path"`
	Level               string `mapstructure:"level"`
	Compress            bool   `mapstructure:"compress"`
	Stat                bool   `mapstructure:"stat"`
	KeepDays            int    `mapstructure:"keep_days"`
	StackCooldownMillis int    `mapstructure:"stack_cooldown_millis"`
	MaxSize             int    `mapstructure:"max_size"`
	Rotation            string `mapstructure:"rotation"`
}

type EnterPrisField struct {
	CertificateName string `mapstructure:"certificateName"`
	IssuerName      string `mapstructure:"issuerName"`
	Operator        string `mapstructure:"operator"`
	LegalName       string `mapstructure:"legalName"`
	LicenceSN       string `mapstructure:"licencesn"`
	EntName         string `mapstructure:"entname"`
	Opscope         string `mapstructure:"opscope"`
	UniscID         string `mapstructure:"uniscid"`
	Dom             string `mapstructure:"dom"`
	OpFrom          string `mapstructure:"opfrom"`
	OpTo            string `mapstructure:"opto"`
}

type PersonField struct {
	CertificateName    string `mapstructure:"certificateName"`
	IssuerName         string `mapstructure:"issuerName"`
	Name               string `mapstructure:"name"`
	Phone              string `mapstructure:"phone"`
	IdentityCardNumber string `mapstructure:"identityCardNumber"`
	CtidVc             string `mapstructure:"ctidVc"`
}

func (log Log) ToLogConf() logx.LogConf {
	return logx.LogConf{
		ServiceName:         log.ServiceName,
		Mode:                log.Mode,
		Encoding:            log.Encoding,
		TimeFormat:          log.TimeFormat,
		Path:                log.Path,
		Level:               log.Level,
		Compress:            log.Compress,
		Stat:                log.Stat,
		KeepDays:            log.KeepDays,
		StackCooldownMillis: log.StackCooldownMillis,
		MaxSize:             log.MaxSize,
		Rotation:            log.Rotation,
	}
}

func ReadTemplate(templateFile string) (*TemplateConfig, error) {
	template := &TemplateConfig{}
	jsonFile, err := os.ReadFile(templateFile)
	if err != nil {
		return nil, err
	}

	_, file := filepath.Split(templateFile)
	templateMap := make(map[string]interface{})
	err = json.Unmarshal(jsonFile, &templateMap)
	if err != nil {
		return nil, err
	}

	template.TemplateId = strings.TrimSuffix(file, ".json")
	template.CredentialSubject = templateMap
	return template, nil
}

/*
1. 读文件，映射到对象

*/

func ReadConfig(path string) (*Config, error) {
	var c = &Config{}
	// 拆分配置文件路径为目录路径和文件名
	dir, file := filepath.Split(path)
	// 设置配置文件的名字
	viper.SetConfigName(file[:len(file)-len(filepath.Ext(file))])
	// 设置配置文件的类型
	viper.SetConfigType("yaml")
	// 添加配置文件的路径，指定 config 目录下寻找
	viper.AddConfigPath(dir)
	// 寻找配置文件并读取
	err := viper.ReadInConfig()
	if err != nil {
		return nil, fmt.Errorf("fatal error config file: %w", err)
	}
	err = viper.Unmarshal(c)
	if err != nil {
		return nil, fmt.Errorf("fatal error unmarshalling config file: %w", err)
	}
	return c, nil
}
