package process

import (
	"cnbn-did/didi10n/config"
	"cnbn-did/didi10n/service"
	"cnbn-did/didi10n/tools"
	"encoding/json"
	"fmt"
	"os"
	"strconv"
	"time"

	"chainmaker.org/chainmaker/common/v2/crypto"
)

const TableNameDid = "did"

// Did DID主表
type Did struct {
	ID          string     `gorm:"column:id;type:varchar(128);primaryKey;comment:DID" json:"id"`                                // DID
	Document    string     `gorm:"column:document;type:text;not null;comment:DID document" json:"document"`                     // DID document
	Txid        string     `gorm:"column:txid;type:varchar(64);not null;comment:DID上链hash" json:"txid"`                         // DID上链hash
	DocumentURL string     `gorm:"column:document_url;type:varchar(255);not null;comment:DID文档保存地址" json:"document_url"`        // DID文档保存地址
	Created     *time.Time `gorm:"column:created;type:datetime;not null;default:CURRENT_TIMESTAMP;comment:创建时间" json:"created"` // 创建时间
	Updated     *time.Time `gorm:"column:updated;type:datetime;not null;default:CURRENT_TIMESTAMP;comment:更新时间" json:"updated"` // 更新时间
	Version     int64      `gorm:"column:version;type:bigint(20);not null;comment:版本号" json:"version"`                          // 版本号
	Black       bool       `gorm:"column:black;type:tinyint(1);not null;comment:黑名单" json:"black"`                              // 黑名单
}

func (*Did) TableName() string {
	return TableNameDid
}

type Integration struct {
	config         config.Config
	templateConfig config.TemplateConfig
	privateKey     crypto.PrivateKey
	did            string
}

func RunGenerationDid(configFile, templateFile string, num int) error {
	runtime, err := service.RuntimeInit(configFile, templateFile)
	if err != nil {
		return err
	}
	instance := service.NewIntegration(runtime)

	var (
		entname string
	)

	if instance.Runtime.Template.TemplateId == service.CompanyTemplateID {
		entname = instance.Runtime.Template.CredentialSubject["entname"].(string) + strconv.FormatInt(time.Now().Unix(), 10)
	}
	for i := 0; i < num; i++ {
		env, t := tools.OutPutEnv(configFile, templateFile)
		date := time.Now().Format(time.RFC3339)

		var title string
		if instance.Runtime.Template.TemplateId == service.CompanyTemplateID {
			instance.Runtime.Template.CredentialSubject["entname"] = entname + "_" + strconv.Itoa(i+1)
			title = fmt.Sprintf("%s_%v", date, instance.Runtime.Template.CredentialSubject["entname"])
		} else {
			title = fmt.Sprintf("%s环境生成%v个%sDID", date, num, t)
		}

		result := fmt.Sprintf("# %v（%s）\n", env, title)

		privateKey, response, doc, err := instance.Register()
		if err != nil {
			return err
		}

		pk, err := privateKey.PublicKey().String()
		if err != nil {
			return err
		}

		sk, err := privateKey.String()
		if err != nil {
			return err
		}

		instance.PrivateKey = privateKey
		instance.Document = doc
		vc, err := instance.GenerateVc(doc, instance.Runtime.Template)
		if err != nil {
			return err
		}

		res, err := instance.DocumentRegister(response, vc, doc)
		if err != nil {
			return err
		}

		documentJson, err := json.MarshalIndent(res.Data.Document, "", "    ")
		if err != nil {
			return err
		}

		vcs, err := instance.GetVcListUser()
		if err != nil {
			return err
		}

		result += fmt.Sprintf("## DID %d\n", i+1)
		result += fmt.Sprintf("DID：\n```\n%s\n```\n", doc.Id)
		result += fmt.Sprintf("公钥：\n```\n%s\n```\n", pk)
		result += fmt.Sprintf("私钥：\n```\n%s\n```\n", sk)
		result += fmt.Sprintf("文档：\n```\n%s\n```\n", documentJson)
		if len(vcs) != 0 {
			m := map[string]interface{}{}
			err := json.Unmarshal([]byte(vcs[0]), &m)
			if err != nil {
				return err
			}
			indent, err := json.MarshalIndent(m, "", "    ")
			if err != nil {
				return err
			}
			result += fmt.Sprintf("凭证：\n```\n%s\n```\n", string(indent))
		}
		err = os.WriteFile(title+".md", []byte(result), 0644)
		if err != nil {
			fmt.Println("Error writing to Markdown file:", err)
			return err
		}

		fmt.Println("Markdown file saved successfully!")

	}

	return nil
}
