/*
Created by guoxin in 2024/1/31 22:48
*/
package service

import (
	"cnbn-did/didi10n/tools"
	"cnbn-did/internal/types"
	"encoding/json"
	"strconv"
	"time"

	"chainmaker.org/chainmaker/common/v2/crypto"
	"chainmaker.org/chainmaker/common/v2/random/uuid"
	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"chainmaker.org/chainmaker/opennet/servicecommon/page"
	"chainweaver.org.cn/chainweaver/did/core"
	corecrypto "chainweaver.org.cn/chainweaver/did/core/crypto"
	"github.com/zeromicro/go-zero/core/logx"
)

func (i *Integration) RegisterOld() (privateKey crypto.PrivateKey, document *core.Document, err error) {
	document = &core.Document{}
	privateKey, sk, pk, err := GenerateKeyPair()
	if err != nil {
		return nil, nil, err
	}

	logx.Info("私钥：\n%v\n", sk)
	logx.Info("公钥：\n%v\n", pk)

	url := i.Runtime.Config.CNBNDID + "/api/v1/did/generate"

	t := &types.DidGenerateRequest{
		PublicKey: []string{
			pk,
		},
		Name:               "袁太富",
		IdentityCardNumber: "510623199206036614",
		Phone:              "18888888888",
	}

	body, err := tools.SendRequest("注册-生成document", url, t, i.Runtime.Config.CNBNDID)
	if err != nil {
		return nil, nil, err
	}
	var result = &vo.DidGenerateResponse{}
	err = json.Unmarshal(body, result)
	if err != nil {
		return nil, nil, err
	}

	err = json.Unmarshal(result.Data.Plaintext[0].Plaintext, document)
	if err != nil {
		return nil, nil, err
	}
	docRequest, err := GenerateRequest(t, result, privateKey, document)
	if err != nil {
		return nil, nil, err
	}

	url = i.Runtime.Config.CNBNDID + "/api/v1/did/register"

	body, err = tools.SendRequest("注册-注册did", url, docRequest, i.Runtime.Config.CNBNDID)
	if err != nil {
		panic(err)
	}
	logx.Info("=======================================注册完成======================================")
	t2 := &vo.VcListRequest{
		Page:       &page.Page{Page: 1, Size: 10},
		Holder:     document.GetId(),
		Issuer:     DefaultIssuer,
		TemplateId: DefaultTemplateID,
		Status:     -1,
	}

	presentation := core.NewVerifiablePresentation(
		core.DefaultVCContext,
		uuid.GetUUID(),
		DefaultIssuer,
		core.VPType,
		"VerifiablePresentation",
		nil,
		nil,
		time.Now().Add(time.Hour*2),
		t2,
	)

	err = corecrypto.NewSigner(privateKey, document.GetId()+"#key-1").Vp(presentation)
	if err != nil {
		return nil, nil, err
	}

	url = i.Runtime.Config.CNBNDID + "/api/v1/did/vc/list"

	body, err = tools.SendRequest("查询vc列表（协议）", url, presentation, i.Runtime.Config.CNBNDID)
	if err != nil {
		return nil, nil, err
	}
	logx.Info("=======================================查询vc列表完成======================================")
	req := &types.DocumentGetRequest{Did: document.GetId()}
	_, err = tools.SendRequest("查询document", i.Runtime.Config.CNBNDID+"/api/v1/did/document/get", req, i.Runtime.Config.CNBNDID)
	if err != nil {
		return
	}
	logx.Info("=======================================查询document完成======================================")
	return
}

// GenerateRequest
//
//	@Description: 根据DidGenerateRequest创建DidRegisterRequest
//	@param req
//	@param result
//	@param privateKey
//	@param doc
//	@return document
//	@return err
func GenerateRequest(req *types.DidGenerateRequest, result *vo.DidGenerateResponse, privateKey crypto.PrivateKey, doc *core.Document) (document *types.DidRegisterRequest, err error) {

	proofs := make([]*vo.LongProof, 0)
	result.Data.Plaintext = append(result.Data.Plaintext, &vo.LongPlaintext{
		Type:      vo.ProofType_VerifyVP,
		Plaintext: []byte(strconv.FormatInt(time.Now().Unix(), 10)),
	})

	for _, plaintext := range result.Data.Plaintext {
		longProof, err := SignLongProof(privateKey, plaintext, doc.Authentication[0])
		if err != nil {
			return nil, err
		}
		proofs = append(proofs, longProof)
	}

	marshal, err := json.Marshal(proofs)
	if err != nil {
		return nil, err
	}

	var registerProof = make([]types.LongProof, 0)
	err = json.Unmarshal(marshal, &registerProof)
	if err != nil {
		return nil, err
	}

	v := &types.DidRegisterRequest{
		Did:                result.Data.Did,
		Name:               req.Name,
		Phone:              req.Phone,
		IdentityCardNumber: req.IdentityCardNumber,
		RegisterProof:      registerProof,
	}

	return v, nil
}

func SignLongProofRandom(privateKey crypto.PrivateKey, authentication string) (proof *vo.LongProof, err error) {
	v := &vo.LongPlaintext{
		Type:      vo.ProofType_VerifyVP,
		Plaintext: []byte(strconv.FormatInt(time.Now().Unix(), 10)),
	}
	return SignLongProof(privateKey, v, authentication)
}

func SignLongProof(privateKey crypto.PrivateKey, plaintext *vo.LongPlaintext, authentication string) (proof *vo.LongProof, err error) {
	coreProof, err := corecrypto.NewSigner(privateKey, authentication).Plaintext(plaintext.Plaintext)
	if err != nil {
		return
	}

	return &vo.LongProof{
		Type: plaintext.Type,
		Proof: &core.SODProof{
			Proof:     coreProof,
			Plaintext: plaintext.Plaintext,
		},
	}, nil
}

const (
	RealNameCertificateName = "长安链个人实名认证证书"
	IssuerName              = "国家区块链创新中心"
	RealNameVCType          = "IdentityCredential"
	RealNameVCTemplateId    = "100000"
)

type BaseCertificate struct {
	Id string `json:"id,omitempty"`
	// 签发机构名称
	IssuerName string `json:"issuerName"`
	// 证书名称
	CertificateName string `json:"certificateName,omitempty"`
}

type RealNameCertificate struct {
	BaseCertificate

	// 用户姓名
	Name string `json:"name,omitempty"`
	// 身份证号
	IdentityCardNumber string `json:"identityCardNumber,omitempty"`
	// 手机号
	Phone string `json:"phone,omitempty"`
	// ctidVc
	CtidVc string `json:"ctidVc,omitempty"`
}

// NewRealNameCertificate
// @Description: 创建实名认证证书
// @param did
// @param name
// @param identityCardNumber
// @param phone
// @param ctidVc
// @return *RealNameCertificate
func NewRealNameCertificate(did, name, identityCardNumber, phone, ctidVc string) *RealNameCertificate {
	return &RealNameCertificate{
		BaseCertificate: BaseCertificate{
			Id:              did,
			IssuerName:      IssuerName,
			CertificateName: RealNameCertificateName,
		},
		Name:               name,
		IdentityCardNumber: identityCardNumber,
		Phone:              phone,
		CtidVc:             ctidVc,
	}
}
