/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package handler

import (
	"chainmaker.org/chainmaker/opennet/servicecommon/health"
	"net/http"

	blacklist "cnbn-did/internal/handler/blacklist"
	did "cnbn-did/internal/handler/did"
	document "cnbn-did/internal/handler/document"
	generate "cnbn-did/internal/handler/generate"
	inner "cnbn-did/internal/handler/inner"
	issuelog "cnbn-did/internal/handler/issuelog"
	trustissuer "cnbn-did/internal/handler/trustissuer"
	vc "cnbn-did/internal/handler/vc"
	vctemplate "cnbn-did/internal/handler/vctemplate"
	verify "cnbn-did/internal/handler/verify"
	"cnbn-did/internal/svc"

	"github.com/zeromicro/go-zero/rest"
)

func RegisterHandlers(server *rest.Server, serverCtx *svc.ServiceContext, version, name, gitCommit, gitBranch, buildTime string) {
	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/register",
				Handler: document.RegisterDocumentHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/update",
				Handler: document.UpdateDocumentHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/get",
				Handler: document.GetDocumentHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/list",
				Handler: document.GetListDocumentHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/get-by-address",
				Handler: document.GetDocumentByAddressHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/document"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/document/import",
				Handler: inner.ImportDocumentHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/document/public-key/add",
				Handler: inner.AddPublicKeyHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/document/public-key/del",
				Handler: inner.DelPublicKeyHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/vctemplate/upgrade",
				Handler: inner.UpdateVcTemplateHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/inner"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/generate",
				Handler: did.GenerateHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/register",
				Handler: did.RegisterHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/register",
				Handler: generate.DocumentRegisterHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/public-key/:action",
				Handler: generate.UpdateDocumentVerificationMethodHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/service/:action",
				Handler: generate.UpdateDocumentServiceHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/cnbn-vp",
				Handler: generate.GenCnbnVpHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/generate"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/issue",
				Handler: vc.VcIssueHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/register",
				Handler: vc.CreateHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/list",
				Handler: vc.VcListHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/list/user",
				Handler: vc.VcListUserHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/revoke",
				Handler: vc.VcRevokeHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/find-did-by-company-name",
				Handler: vc.FindDIDByCompanyNameHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/find-company-name-by-did",
				Handler: vc.FindCompanyNameByDIDHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/vc"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/create",
				Handler: vctemplate.CreatevctemplateHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/create/vp",
				Handler: vctemplate.CreatevctemplatevpHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/onequeryvctemplate",
				Handler: vctemplate.OnequeryvctemplateHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/list",
				Handler: vctemplate.PagequeryvctemplateHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/upgrade/vp",
				Handler: vctemplate.UpgradevctemplatevpHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/vctemplate"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/vp",
				Handler: verify.VerifyVpHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/verify"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/add",
				Handler: trustissuer.AddHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/delete",
				Handler: trustissuer.DelHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/list",
				Handler: trustissuer.ListHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/trust-issuer"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/list",
				Handler: issuelog.GetVcIssueLogsHandler(serverCtx),
			},
			{
				Method:  http.MethodPost,
				Path:    "/issuers",
				Handler: issuelog.GetVcIssuersHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/issue-log"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/check",
				Handler: blacklist.CheckHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/blacklist"),
	)

	server.AddRoutes(
		[]rest.Route{
			{
				Method:  http.MethodPost,
				Path:    "/checklist",
				Handler: vc.CheckListHandler(serverCtx),
			},
		},
		rest.WithPrefix("/api/v1/did/vc/revoke"),
	)
	healthRoutes := health.GetRouteWithParams(version, name, gitCommit, gitBranch, buildTime)
	server.AddRoutes(healthRoutes)
}
