package common

import (
	"cnbn-did/errorcode"
	"context"
	"encoding/json"

	"google.golang.org/grpc"
)

type Service[IReq, IPbReq, IPbResp, IResp any] interface {
	ConvertRequestToPb(input *IReq) (*IPbReq, error)
	CallService(ctx context.Context, input *IPbReq, opts ...grpc.CallOption) (*IPbResp, error)
	ConvertResponseToOutput(input *IPbResp) (*IResp, error)
}

type BaseService[IReq, IPbReq, IPbResp, IResp any] struct {
	Service[IReq, IPbReq, IPbResp, IResp]
	Operation func(ctx context.Context, in *IPbReq, opts ...grpc.CallOption) (*IPbResp, error)
}

// ConvertRequestToPb 方法，将IReq转换为IPbReq
func (b *BaseService[IReq, IPbReq, IPbResp, IResp]) ConvertRequestToPb(input *IReq) (*IPbReq, error) {
	var resp *IPbReq
	data, err := json.Marshal(input)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("ConvertRequestToPb input").BuildResult(resp)
		return resp, nil
	}
	var pbRequest *IPbReq
	err = json.Unmarshal(data, &pbRequest)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("ConvertRequestToPb IPbReq").BuildResult(resp)
		return resp, nil
	}
	return pbRequest, nil
}

func (b *BaseService[IReq, IPbReq, IPbResp, IResp]) ConvertResponseToOutput(input *IPbResp) (*IResp, error) {
	var resp *IResp
	data, err := json.Marshal(input)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("ConvertResponseToOutput input").BuildResult(resp)
		return resp, nil
	}
	var response *IResp
	err = json.Unmarshal(data, &response)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("ConvertResponseToOutput").BuildResult(resp)
		return resp, nil
	}
	return response, nil
}
func (b *BaseService[IReq, IPbReq, IPbResp, IResp]) CallService(ctx context.Context, input *IPbReq, opts ...grpc.CallOption) (*IPbResp, error) {
	return b.Operation(ctx, input, opts...)
}

func HandleRequest[IReq, IPbReq, IPbResp, IResp any](ctx context.Context, s Service[IReq, IPbReq, IPbResp, IResp], input *IReq, opts ...grpc.CallOption) (*IResp, error) {
	// 将IReq转换为IPbReq
	var resp *IResp
	pbRequest, err := s.ConvertRequestToPb(input)
	if err != nil {
		errorcode.CommonConvertRequestToPbFailed.AddError(err).AddMessage("input").BuildResult(resp)
		return resp, nil
	}
	// 调用服务
	pbResponse, err := s.CallService(ctx, pbRequest, opts...)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("CallService").BuildResult(resp)
		return resp, nil
	}
	// 将IPbResp转换为IResp
	output, err := s.ConvertResponseToOutput(pbResponse)
	if err != nil {
		errorcode.CommonConvertRequestToPbFailed.AddError(err).AddMessage("pbResponse").BuildResult(resp)
		return resp, nil
	}
	// 返回IResp
	return output, nil
}
