/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package did

import (
	"cnbn-did/errorcode"
	"cnbn-did/internal/logic/common"
	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"
	"context"
	"encoding/json"

	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"google.golang.org/grpc"

	"github.com/zeromicro/go-zero/core/logx"
)

type GenerateLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func (l *GenerateLogic) ConvertRequestToPb(req *types.DidGenerateRequest) (*vo.DidGenerateRequest, error) {
	resp := &vo.DidGenerateRequest{}
	bytes, err := json.Marshal(req)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("DidGenerateRequest").BuildResult(resp)
		return resp, nil
	}

	in := &vo.DidGenerateRequest{
		Type: []vo.ProofType{
			vo.ProofType_RegisterDocument,
			vo.ProofType_CtidBindCert,
			vo.ProofType_CtidRealNameVC,
		},
		Document: &vo.DidGenerateDocument{
			PublicKey: req.PublicKey,
		},
		CtidDID: &vo.DidGenerateCtid{
			PublicKey:          req.PublicKey[0],
			Name:               req.Name,
			IdentityCardNumber: req.IdentityCardNumber,
		},
	}

	err = json.Unmarshal(bytes, in)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("DidGenerateRequest").BuildResult(resp)
		return resp, nil
	}
	return in, nil
}

func (l *GenerateLogic) CallService(ctx context.Context, input *vo.DidGenerateRequest, opts ...grpc.CallOption) (*vo.DidGenerateResponse, error) {
	return l.svcCtx.DidService.Generate(ctx, input, opts...)
}

func (l *GenerateLogic) ConvertResponseToOutput(out *vo.DidGenerateResponse) (*types.DidGenerateResponse, error) {
	var resp *types.DidGenerateResponse
	bytes, err := json.Marshal(out)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("DidGenerateResponse").BuildResult(resp)
		return resp, nil
	}

	err = json.Unmarshal(bytes, &resp)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("DidGenerateResponse").BuildResult(resp)
		return resp, nil
	}
	return resp, nil
}

var _ common.Service[types.DidGenerateRequest, vo.DidGenerateRequest, vo.DidGenerateResponse, types.DidGenerateResponse] = (*GenerateLogic)(nil)

func NewGenerateLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GenerateLogic {
	return &GenerateLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *GenerateLogic) Generate(req *types.DidGenerateRequest) (resp *types.DidGenerateResponse, err error) {

	return common.HandleRequest[types.DidGenerateRequest, vo.DidGenerateRequest, vo.DidGenerateResponse, types.DidGenerateResponse](l.ctx, l, req)
}
