/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package did

import (
	"chainmaker.org/chainmaker/opennet/servicecommon/res"
	"chainmaker.org/chainmaker/opennet/servicecommon/res/code"
	"chainweaver.org.cn/chainweaver/did/core"
	"cnbn-did/errorcode"
	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"
	"context"
	"encoding/json"
	"errors"
	"time"

	vo "chainmaker.org/chainmaker/opennet/did/vo/did"

	"github.com/zeromicro/go-zero/core/logx"
)

type RegisterLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewRegisterLogic(ctx context.Context, svcCtx *svc.ServiceContext) *RegisterLogic {
	return &RegisterLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *RegisterLogic) Register(req *types.DidRegisterRequest) (resp *types.DidRegisterResponse, err error) {
	resp = &types.DidRegisterResponse{}
	if len(req.RegisterProof) != 4 {
		errorcode.InputException.AddError(errors.New("proof length must be 4")).AddMessage("Register").BuildResult(resp)
		return resp, nil
	}
	var (
		proofs   []types.LongProof
		vcProofs []*vo.LongProof
	)

	for _, proof := range req.RegisterProof {
		if proof.Type == 2 || proof.Type == 3 {
			var vcProof = &vo.LongProof{}
			marshal, err := json.Marshal(proof)
			if err != nil {
				errorcode.CommonMarshalException.AddError(err).AddMessage("RegisterProof").BuildResult(resp)
				return resp, nil
			}

			err = json.Unmarshal(marshal, vcProof)
			if err != nil {
				errorcode.CommonUnmarshalException.AddError(err).AddMessage("RegisterProof").BuildResult(resp)
				return resp, nil
			}

			vcProofs = append(vcProofs, vcProof)
		} else {
			proofs = append(proofs, proof)
		}
	}

	req.RegisterProof = proofs

	bytes, err := json.Marshal(req)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("DidRegisterRequest").BuildResult(resp)
		return resp, nil
	}

	in := &vo.DidRegisterRequest{}
	err = json.Unmarshal(bytes, in)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("DidRegisterRequest").BuildResult(resp)
		return resp, nil
	}

	out, err := l.svcCtx.DidService.Register(l.ctx, in)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Register").BuildResult(resp)
		return resp, nil
	}

	if code.CodeOk.NotEqualsInt32(out.Code) {
		res.ErrorMessageCode(code.Code(out.Code), out.Msg).BuildResult(resp)
		return resp, nil
	}

	document := out.Data.Document
	out.Data.Document = nil
	bytes, err = json.Marshal(out)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("DidRegisterResponse").BuildResult(resp)
		return resp, nil
	}

	err = json.Unmarshal(bytes, resp)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("DidRegisterResponse").BuildResult(resp)
		return resp, nil
	}

	err = json.Unmarshal(document, &resp.Data.Document)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("Document").BuildResult(resp)
		return resp, nil
	}

	certificate := NewRealNameCertificate(in.Did, req.Name, req.IdentityCardNumber, req.Phone, "暂无")
	bytes, err = json.Marshal(certificate)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("RealNameCertificate").BuildResult(resp)
		return resp, nil
	}

	now := time.Now()
	issueRequest := &vo.VcIssueRequest{
		TemplateId:        "100000",
		ExpirationDate:    core.Format(now),
		IssuanceDate:      core.Format(now.AddDate(3, 0, 0)),
		CredentialSubject: bytes,
		Proof:             vcProofs,
	}
	issue, err := l.svcCtx.VcService.Issue(l.ctx, issueRequest)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Issue").BuildResult(resp)
		return resp, nil
	}

	if code.CodeOk.NotEqualsInt32(issue.Code) {
		res.ErrorMessageCode(code.Code(issue.Code), issue.Msg).BuildResult(resp)
		return resp, nil
	}

	err = json.Unmarshal([]byte(issue.Data.Vc), &resp.Data.Vc)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage(" DidRegisterResponseDat Vc").BuildResult(resp)
		return resp, nil
	}

	return
}

const (
	RealNameCertificateName = "长安链个人实名认证证书"
	IssuerName              = "国家区块链创新中心"
	RealNameVCType          = "IdentityCredential"
	RealNameVCTemplateId    = "100000"
)

type BaseCertificate struct {
	Id string `json:"id,omitempty"`
	// 签发机构名称
	IssuerName string `json:"issuerName,omitempty"`
	// 证书名称
	CertificateName string `json:"certificateName,omitempty"`
}

type RealNameCertificate struct {
	BaseCertificate

	// 用户姓名
	Name string `json:"name,omitempty"`
	// 身份证号
	IdentityCardNumber string `json:"identityCardNumber,omitempty"`
	// 手机号
	Phone string `json:"phone,omitempty"`
	// ctidVc
	CtidVc string `json:"ctidVc,omitempty"`
}

// NewRealNameCertificate
// @Description: 创建实名认证证书
// @param did
// @param name
// @param identityCardNumber
// @param phone
// @param ctidVc
// @return *RealNameCertificate
func NewRealNameCertificate(did, name, identityCardNumber, phone, ctidVc string) *RealNameCertificate {
	return &RealNameCertificate{
		BaseCertificate: BaseCertificate{
			IssuerName:      IssuerName,
			CertificateName: RealNameCertificateName,
		},
		Name:               name,
		IdentityCardNumber: identityCardNumber,
		Phone:              phone,
		CtidVc:             ctidVc,
	}
}
