/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package document

import (
	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"chainmaker.org/chainmaker/opennet/servicecommon/copier"
	"chainmaker.org/chainmaker/opennet/servicecommon/res"
	"chainmaker.org/chainmaker/opennet/servicecommon/res/code"
	"cnbn-did/errorcode"
	"context"
	"encoding/json"

	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"

	"github.com/zeromicro/go-zero/core/logx"
)

type RegisterDocumentLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewRegisterDocumentLogic(ctx context.Context, svcCtx *svc.ServiceContext) *RegisterDocumentLogic {
	return &RegisterDocumentLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *RegisterDocumentLogic) RegisterDocument(req *types.DocumentRegisterRequest) (*types.DocumentRegisterResponse, error) {
	var (
		in   = &vo.DocumentRegisterRequest{}
		resp = &types.DocumentRegisterResponse{}
	)

	err := copier.Copy(req, in)
	if err != nil {
		errorcode.CommonRequestCopyFailed.AddError(err).AddMessage("DocumentRegisterRequest").BuildResult(resp)
		return resp, nil
	}
	out, err := l.svcCtx.DocumentService.Register(l.ctx, in)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Register").BuildResult(resp)
		return resp, nil
	}
	if code.CodeOk.NotEqualsInt32(out.Code) {
		res.ErrorMessageCode(code.Code(out.Code), out.Msg).BuildResult(resp)
		return resp, nil
	}
	document := types.Document{}
	err = copier.Copy(out.Data.Document, &document)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("Document").BuildResult(resp)
		return resp, nil
	}

	if req.Issue != nil {
		bytes, err := json.Marshal(req.Issue.CredentialSubject)
		if err != nil {
			errorcode.CommonMarshalException.AddError(err).AddMessage("CredentialSubject").BuildResult(resp)
			return resp, nil
		}
		issueReq := &vo.VcIssueRequest{
			TemplateId:        req.Issue.TemplateId,
			Holder:            document.Id,
			ExpirationDate:    req.Issue.ExpirationDate,
			IssuanceDate:      req.Issue.IssuanceDate,
			CredentialSubject: bytes,
		}
		if len(req.Issue.Proof) != 0 {
			issueReq.Proof = make([]*vo.LongProof, 1)
			err = copier.Copy(req.Issue.Proof, &issueReq.Proof)
			if err != nil {
				return nil, err
			}
		}

		issueRes, err := l.svcCtx.VcService.Issue(l.ctx, issueReq)
		if err != nil {
			errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Issue").BuildResult(resp)
			return resp, nil
		}
		if code.CodeOk.NotEqualsInt32(issueRes.Code) {
			res.ErrorMessageCode(code.Code(issueRes.Code), issueRes.Msg).BuildResult(resp)
			return resp, nil
		}
		vc := types.VerifiableCredential{}
		err = json.Unmarshal([]byte(issueRes.Data.Vc), &vc)
		if err != nil {
			errorcode.CommonUnmarshalException.AddError(err).AddMessage("Issue").BuildResult(resp)
			return resp, nil
		}
		res.OK.BuildResultData(resp, types.DocumentRegisterResponseData{Document: document, Vc: &vc})
		return resp, nil

	}
	res.OK.BuildResultData(resp, types.DocumentRegisterResponseData{Document: document})
	return resp, nil
}
