/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package vc

import (
	"chainmaker.org/chainmaker/opennet/servicecommon/res/code"
	"cnbn-did/errorcode"
	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"
	"context"
	"encoding/json"

	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"chainmaker.org/chainmaker/opennet/servicecommon/copier"
	"chainmaker.org/chainmaker/opennet/servicecommon/res"
	"chainweaver.org.cn/chainweaver/did/core"
	"github.com/zeromicro/go-zero/core/logx"
)

type CreateLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewCreateLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreateLogic {
	return &CreateLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *CreateLogic) Create(req *types.VerifiablePresentation) (resp *types.IssueLogCreateResponse, err error) {
	vp := &core.VerifiablePresentation{}
	resp = &types.IssueLogCreateResponse{}
	err = copier.Copy(&req, &vp)
	if err != nil {
		errorcode.InputException.AddError(err).AddMessage("CreateIssueLog").BuildResult(resp)
		return resp, nil
	}

	verifyReq := &vo.VpRequestWrapper{
		Vp: vp,
	}
	response, err := l.svcCtx.VerifyService.Vp(l.ctx, verifyReq)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("CreateIssueLog中VerifyVp").BuildResult(resp)
		return resp, nil
	}

	if code.CodeOk.NotEqualsInt32(response.Code) {
		err = copier.Copy(&response, &resp)
		if err != nil {
			errorcode.ServiceResultException.AddError(err).AddMessage("CreateIssueLog中VerifyVp").BuildResult(resp)
			return resp, nil
		}
		return
	}

	vcIssueLogReq := &vo.VcIssueLogRequest{}
	err = json.Unmarshal(vp.Extend, vcIssueLogReq)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("CreateIssueLog").BuildResult(resp)
		return resp, nil
	}

	result, err := l.svcCtx.VcIssuerLogService.VcIssuerLog(l.ctx, vcIssueLogReq)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("CreateIssueLog").BuildResult(resp)
		return resp, nil
	}

	resp = new(types.IssueLogCreateResponse)
	if code.CodeOk.EqualsInt32(result.Code) {
		err = copier.Copy(&result, &resp)
		if err != nil {
			errorcode.ServiceResultException.AddError(err).AddMessage("CreateIssueLog").BuildResult(resp)
			return resp, nil
		}
		return resp, nil

	} else {
		res.ErrorMessageCode(code.Code(result.Code), result.Msg).AddMessage("CreateIssueLog code is not 200000").BuildResult(resp)
		return resp, nil
	}

	return
}
