/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package vctemplate

import (
	"chainmaker.org/chainmaker/opennet/servicecommon/res/code"
	"cnbn-did/errorcode"
	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"
	"context"
	"encoding/json"

	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"chainmaker.org/chainmaker/opennet/servicecommon/copier"
	"chainmaker.org/chainmaker/opennet/servicecommon/res"
	"chainweaver.org.cn/chainweaver/did/core"
	"github.com/zeromicro/go-zero/core/logx"
)

type CreatevctemplatevpLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewCreatevctemplatevpLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreatevctemplatevpLogic {
	return &CreatevctemplatevpLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

type Extend struct {
	Name                 string                 `json:"name"`
	Versions             string                 `json:"versions"`
	Creator              string                 `json:"creator"`
	VcType               string                 `json:"vcType"`
	Properties           map[string]PropertyObj `json:"properties"`
	Required             []string               `json:"required"`
	AdditionalProperties bool                   `json:"additionalProperties"`
}

type PropertyObj struct {
	Type      string   `json:"type"`
	Title     string   `json:"title"`
	Pattern   string   `json:"pattern,omitempty,optional"`
	Desc      string   `json:"desc,omitempty,optional"`
	Enum      []string `json:"enum,omitempty,optional"`      // 用于枚举类型的值
	Minimum   *float64 `json:"minimum,omitempty,optional"`   // 用于数字类型的最小值
	Maximum   *float64 `json:"maximum,omitempty,optional"`   // 用于数字类型的最大值
	MinLength *int     `json:"minLength,omitempty,optional"` // 用于字符串类型的最小长度
	MaxLength *int     `json:"maxLength,omitempty,optional"` // 用于字符串类型的最大长度
	Format    string   `json:"format,omitempty,optional"`    // 用于字符串类型的格式（如日期、电子邮件等）
}

func (l *CreatevctemplatevpLogic) Createvctemplatevp(req *types.VerifiablePresentation) (resp *types.VcTemplateSaveResponse, err error) {
	vp := &core.VerifiablePresentation{}
	resp = &types.VcTemplateSaveResponse{}
	err = copier.Copy(&req, &vp)
	if err != nil {
		errorcode.InputException.AddError(err).AddMessage("Createvctemplate").BuildResult(resp)
		return resp, nil
	}

	verifyReq := &vo.VpRequestWrapper{
		Vp: vp,
	}
	response, err := l.svcCtx.VerifyService.Vp(l.ctx, verifyReq)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Createvctemplate中VerifyVp").BuildResult(resp)
		return resp, nil
	}

	if code.CodeOk.NotEqualsInt32(response.Code) {
		err = copier.Copy(&response, &resp)
		if err != nil {
			errorcode.ServiceResultException.AddError(err).AddMessage("Createvctemplate中VerifyVp").BuildResult(resp)
			return resp, nil
		}
		return
	}

	var extend Extend
	schema := string(vp.Extend)
	err = json.Unmarshal(vp.Extend, &extend)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("Createvctemplate中extend").BuildResult(resp)
		return resp, nil
	}

	creatReq := &vo.VcTemplateSaveRequest{
		Name:     extend.Name,
		Versions: extend.Versions,
		Creator:  extend.Creator,
		VcType:   extend.VcType,
		Schema:   schema,
	}

	result, err := l.svcCtx.VcTemplate.Create(l.ctx, creatReq)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Createvctemplate").BuildResult(resp)
		return resp, nil
	}
	resp = new(types.VcTemplateSaveResponse)
	if code.CodeOk.EqualsInt32(result.Code) {
		err = copier.Copy(&result, &resp)
		if err != nil {
			errorcode.ServiceResultException.AddError(err).AddMessage("Createvctemplate").BuildResult(resp)
			return resp, nil
		}
		return resp, nil

	} else {
		res.ErrorMessageCode(code.Code(result.Code), result.Msg).AddMessage("Createvctemplate code is not 200000").BuildResult(resp)
		return resp, nil
	}
	return
}
