/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package vctemplate

import (
	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"chainmaker.org/chainmaker/opennet/servicecommon/copier"
	"chainmaker.org/chainmaker/opennet/servicecommon/res"
	"chainmaker.org/chainmaker/opennet/servicecommon/res/code"
	"chainweaver.org.cn/chainweaver/did/core"
	"cnbn-did/errorcode"
	"context"
	"encoding/json"

	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"

	"github.com/zeromicro/go-zero/core/logx"
)

type UpgradevctemplatevpLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func NewUpgradevctemplatevpLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UpgradevctemplatevpLogic {
	return &UpgradevctemplatevpLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

type ExtendByUpgrade struct {
	Id                   string                 `json:"id"`
	Name                 string                 `json:"name"`
	Versions             string                 `json:"versions"`
	Creator              string                 `json:"creator"`
	VcType               string                 `json:"vcType"`
	Properties           map[string]PropertyObj `json:"properties"`
	Required             []string               `json:"required"`
	AdditionalProperties bool                   `json:"additionalProperties"`
}

func (l *UpgradevctemplatevpLogic) Upgradevctemplatevp(req *types.VerifiablePresentation) (resp *types.VcTemplateUpgradeResponse, err error) {
	vp := &core.VerifiablePresentation{}
	resp = &types.VcTemplateUpgradeResponse{}
	err = copier.Copy(&req, &vp)
	if err != nil {
		errorcode.InputException.AddError(err).AddMessage("Upgradevctemplatevp").BuildResult(resp)
		return resp, nil
	}

	verifyReq := &vo.VpRequestWrapper{
		Vp: vp,
	}
	response, err := l.svcCtx.VerifyService.Vp(l.ctx, verifyReq)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Upgradevctemplate中VerifyVp").BuildResult(resp)
		return resp, nil
	}

	if code.CodeOk.NotEqualsInt32(response.Code) {
		err = copier.Copy(&response, &resp)
		if err != nil {
			errorcode.ServiceResultException.AddError(err).AddMessage("Createvctemplate中VerifyVp").BuildResult(resp)
			return resp, nil
		}
		return
	}

	var extend ExtendByUpgrade
	err = json.Unmarshal(vp.Extend, &extend)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("Upgradevctemplate中extend").BuildResult(resp)
		return resp, nil
	}

	// TODO 复用不带vp的入参
	// 移除 extend 中的 id 字段
	var extendMap map[string]interface{}
	err = json.Unmarshal(vp.Extend, &extendMap)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("Upgradevctemplate中extendMap").BuildResult(resp)
		return resp, nil
	}
	delete(extendMap, "id")

	// 将结果重新编码为 JSON 字符串
	schemaBytes, err := json.Marshal(extendMap)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("Upgradevctemplate中schemaBytes").BuildResult(resp)
		return resp, nil
	}
	schema := string(schemaBytes)

	upgradeReq := &vo.VcTemplateUpgradeRequest{
		Id:       extend.Id,
		Name:     extend.Name,
		Versions: extend.Versions,
		Creator:  extend.Creator,
		VcType:   extend.VcType,
		Schema:   schema,
	}

	result, err := l.svcCtx.VcTemplate.Upgrade(l.ctx, upgradeReq)
	if err != nil {
		errorcode.CommonCallServiceFailed.AddError(err).AddMessage("Upgradevctemplatevp").BuildResult(resp)
		return resp, nil
	}
	resp = new(types.VcTemplateUpgradeResponse)
	if code.CodeOk.EqualsInt32(result.Code) {
		err = copier.Copy(&result, &resp)
		if err != nil {
			errorcode.ServiceResultException.AddError(err).AddMessage("Upgradevctemplatevp").BuildResult(resp)
			return resp, nil
		}
		return resp, nil

	} else {
		res.ErrorMessageCode(code.Code(result.Code), result.Msg).AddMessage("Upgradevctemplatevp code is not 200000").BuildResult(resp)
		return resp, nil
	}
	return
}
