/*
Copyright (C) BABEC. All rights reserved.

SPDX-License-Identifier: Apache-2.0
*/

package verify

import (
	vo "chainmaker.org/chainmaker/opennet/did/vo/did"
	"chainweaver.org.cn/chainweaver/did/core"
	"cnbn-did/errorcode"
	"cnbn-did/internal/logic/common"
	"context"
	"encoding/json"
	"google.golang.org/grpc"

	"cnbn-did/internal/svc"
	"cnbn-did/internal/types"

	"github.com/zeromicro/go-zero/core/logx"
)

var _ common.Service[types.VerifiablePresentation, vo.VpRequestWrapper, vo.VerifyResponse, types.VerifyResponse] = (*VerifyVpLogic)(nil)

type VerifyVpLogic struct {
	logx.Logger
	ctx    context.Context
	svcCtx *svc.ServiceContext
}

func (l *VerifyVpLogic) ConvertRequestToPb(req *types.VerifiablePresentation) (*vo.VpRequestWrapper, error) {
	resp := &vo.VpRequestWrapper{}
	vp, err := json.Marshal(req)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("VerifiablePresentation").BuildResult(resp)
		return resp, nil
	}
	coreVp := &core.VerifiablePresentation{}
	err = json.Unmarshal(vp, coreVp)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("Verify vp").BuildResult(resp)
		return resp, nil
	}
	in := &vo.VpRequestWrapper{
		Vp: coreVp,
	}
	return in, nil
}

func (l *VerifyVpLogic) CallService(ctx context.Context, req *vo.VpRequestWrapper, opts ...grpc.CallOption) (*vo.VerifyResponse, error) {
	return l.svcCtx.VerifyService.Vp(ctx, req, opts...)
}

func (l *VerifyVpLogic) ConvertResponseToOutput(req *vo.VerifyResponse) (*types.VerifyResponse, error) {
	var resp *types.VerifyResponse
	bytes, err := json.Marshal(req)
	if err != nil {
		errorcode.CommonMarshalException.AddError(err).AddMessage("VerifyResponse").BuildResult(resp)
		return resp, nil
	}

	err = json.Unmarshal(bytes, &resp)
	if err != nil {
		errorcode.CommonUnmarshalException.AddError(err).AddMessage("VerifyResponse").BuildResult(resp)
		return resp, nil
	}
	return resp, nil
}

func NewVerifyVpLogic(ctx context.Context, svcCtx *svc.ServiceContext) *VerifyVpLogic {
	return &VerifyVpLogic{
		Logger: logx.WithContext(ctx),
		ctx:    ctx,
		svcCtx: svcCtx,
	}
}

func (l *VerifyVpLogic) VerifyVp(req *types.VerifiablePresentation) (resp *types.VerifyResponse, err error) {
	return common.HandleRequest[types.VerifiablePresentation, vo.VpRequestWrapper, vo.VerifyResponse, types.VerifyResponse](l.ctx, l, req)
}
