// Copyright (C) BABEC. All rights reserved.
// Copyright (C) THL A29 Limited, a Tencent company. All rights reserved.
//
// SPDX-License-Identifier: Apache-2.0

package core

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"chainmaker.org/chainmaker/pb-go/v2/common"

	"chainweaver.org.cn/chainweaver/did/did-sync/core/handle"
	"chainweaver.org.cn/chainweaver/did/did-sync/core/listen"
	"chainweaver.org.cn/chainweaver/did/did-sync/core/write"
	"chainweaver.org.cn/chainweaver/did/did-sync/db/sqlite"
	"chainweaver.org.cn/chainweaver/did/did-sync/global"
	"chainweaver.org.cn/chainweaver/did/did-sync/initialize"
)

var (
	getBlockHeight = "GetBlockHeight"
)

type SyncHandle struct {
	ctx       context.Context
	errorC    chan error
	Listener  *listen.EventListener
	Handler   *handle.BlockHandler
	Committer *write.EventsCommit
}

// NewSyncHandle new SyncHandle
func NewSyncHandle(ctx context.Context, errorC chan error) (error, *SyncHandle) {
	contractName := global.CONFIG.SubCmSdk.ContractName

	var startBlock int64
	// 从子链 同步合约 获取区块高度
	for {
		resp, err := initialize.SubSdkClient.SdkClient.QueryContract(contractName, getBlockHeight, nil, -1)
		if err != nil {
			global.LOG.Errorf("The Client failed to query the contract: %s", err)
			time.Sleep(5 * time.Minute)
			continue
		}
		if resp.Code == common.TxStatusCode_SUCCESS {
			blockHeight, _ := strconv.ParseUint(string(resp.ContractResult.Result), 10, 64)
			startBlock = int64(blockHeight)
			global.LOG.Infof("sync start height get from contract is: %d", startBlock)
		}
		break
	}

	// 从数据库查询
	state, err := sqlite.GetState(global.CONFIG.SubCmSdk.ChainConfig.ChainId)
	if err != nil && err.Error() == "record not found" {
		state, err = initState(global.CONFIG.EventContract.StartHeight)
	} else if err != nil {
		return err, nil
	}
	if state.SyncHeight > startBlock {
		startBlock = state.SyncHeight
		global.LOG.Info(fmt.Sprintf("sync start height in sql is: %d", startBlock))
	}

	if global.CONFIG.EventContract.StartHeight > startBlock {
		startBlock = global.CONFIG.EventContract.StartHeight
		global.LOG.Info(fmt.Sprintf("sync start height in config is: %d", startBlock))
	}

	validTopic := map[string]struct{}{}
	for _, topic := range global.CONFIG.EventContract.Topic {
		validTopic[topic] = struct{}{}
	}

	listener := listen.NewEventListener(startBlock, global.CONFIG.EventContract.EndHeight, errorC)
	handler := handle.NewBlockHandler(listener.BlockChan, validTopic, errorC)
	committer := write.NewBlockHandler(handler.EventChan, errorC)

	return nil, &SyncHandle{
		ctx:       ctx,
		errorC:    errorC,
		Listener:  listener,
		Handler:   handler,
		Committer: committer,
	}
}

// StartSync start sync service
func (sync *SyncHandle) StartSync() error {
	err := sync.Listener.Start(sync.ctx)
	if err != nil {
		return err
	}

	sync.Handler.Start(sync.ctx)
	sync.Committer.Start(sync.ctx)
	return nil
}

// Close close
func (sync *SyncHandle) Close() {
	//todo
}

func initState(startHeight int64) (*sqlite.State, error) {
	state := sqlite.State{
		ChainType:  global.CONFIG.SubCmSdk.ChainConfig.ChainId,
		SyncHeight: startHeight,
	}

	return state.InsertStatus()
}
