// Copyright (C) BABEC. All rights reserved.
// Copyright (C) THL A29 Limited, a Tencent company. All rights reserved.
//
// SPDX-License-Identifier: Apache-2.0

package handle

import (
	"context"
	"fmt"
	"strconv"

	"chainmaker.org/chainmaker/pb-go/v2/common"

	com "chainweaver.org.cn/chainweaver/did/did-sync/common"
	"chainweaver.org.cn/chainweaver/did/did-sync/global"
	"chainweaver.org.cn/chainweaver/did/did-sync/initialize"
)

var (
	setBlockHeight = "SetBlockHeight"
)

// Transaction dependency in adjacency table representation
type dagNeighbors map[int]struct{}

// BlockHandler contract event listener
type BlockHandler struct {
	BlockChan  <-chan *common.BlockInfo
	validTopic map[string]struct{}
	EventChan  chan *com.BlockEvent
	errorC     chan error
}

// NewBlockHandler new BlockHandler
func NewBlockHandler(block <-chan *common.BlockInfo, validTopic map[string]struct{}, errorC chan error) *BlockHandler {
	return &BlockHandler{
		BlockChan:  block,
		validTopic: validTopic,
		EventChan:  make(chan *com.BlockEvent),
		errorC:     errorC,
	}
}

// Start get events from block and add to loop
func (e *BlockHandler) Start(ctx context.Context) error {
	go e.loop(ctx, e.EventChan)
	return nil
}

// Close event subscribe close
func (e *BlockHandler) Close(err error) {
	global.LOG.Error(err.Error())
}

func (e *BlockHandler) loop(ctx context.Context, eventChan chan *com.BlockEvent) {
	var num int
	for {
		select {
		case block, ok := <-e.BlockChan:
			if !ok {
				global.LOG.Error("chan is close!")
				return
			}

			events, err := e.getValidEvents(block)
			if err != nil {
				global.LOG.Error(fmt.Sprintf("get valid events err[%s]", err))
				e.errorC <- fmt.Errorf("get valid events err[%s]", err)
				return
			}

			if len(events) != 0 {
				eventChan <- &com.BlockEvent{Height: block.Block.Header.BlockHeight, Events: events}
			}

			num++
			if num == 100 {
				str := strconv.FormatUint(block.Block.Header.BlockHeight-1, 10)
				setBlockHeightKv := []*common.KeyValuePair{
					{
						Key:   "blockHeight",
						Value: []byte(str),
					},
				}
				global.LOG.Debugf("invoke contract, [contractName:%s]/[method:%s]/[kvs:%v]",
					global.CONFIG.SubCmSdk.ContractName, setBlockHeight, setBlockHeightKv)
				resp, err := initialize.SubSdkClient.SdkClient.InvokeContract(
					global.CONFIG.SubCmSdk.ContractName, setBlockHeight, "", setBlockHeightKv, -1, true)
				if err != nil {
					return
				}

				if resp.Code != common.TxStatusCode_SUCCESS {
					return
				}

				global.LOG.Infof("invoke contract success, resp: [code:%d]/[msg:%s]/"+
					"[contractResult:%s]", resp.Code, resp.Message, resp.ContractResult)
				num = 0
			}

		case <-ctx.Done():
			return
		}
	}
}

func (e *BlockHandler) getValidEvents(block *common.BlockInfo) ([][]*common.ContractEvent, error) {
	events := make(map[int][]*common.ContractEvent)
	var resultEvent [][]*common.ContractEvent
	for i, tx := range block.Block.Txs {
		if tx.Result.Code == common.TxStatusCode_SUCCESS {
			events[i] = tx.Result.ContractResult.ContractEvent
		}
	}

	txIndexBatch, dagRemain, reverseDagRemain, err := initSimulateDag(block.Block.Dag)
	if err != nil {
		return nil, err
	}

	for _, i := range txIndexBatch {
		var validEvents []*common.ContractEvent
		if events[i] == nil {
			continue
		}
		for _, event := range events[i] {
			if event.ContractName == global.CONFIG.EventContract.Name {
				if len(e.validTopic) == 0 {
					validEvents = append(validEvents, event)
				} else if _, ok := e.validTopic[event.Topic]; ok {
					validEvents = append(validEvents, event)
				}
			}
		}
		if len(validEvents) != 0 {
			resultEvent = append(resultEvent, validEvents)
		}
		txIndexBatch = append(txIndexBatch, shrinkDag(i, dagRemain, reverseDagRemain)...)
	}

	return resultEvent, nil
}

// initSimulateDag init simulate dag
func initSimulateDag(dag *common.DAG) (
	[]int, map[int]dagNeighbors, map[int]dagNeighbors, error) {
	dagRemain := make(map[int]dagNeighbors, len(dag.Vertexes))
	reverseDagRemain := make(map[int]dagNeighbors, len(dag.Vertexes)*4)
	var txIndexBatch []int // 存放的是没有相邻关系的交易
	for txIndex, neighbors := range dag.Vertexes {
		if neighbors == nil {
			return nil, nil, nil, fmt.Errorf("dag has nil neighbor")
		}
		if len(neighbors.Neighbors) == 0 {
			txIndexBatch = append(txIndexBatch, txIndex)
			continue
		}
		dn := make(dagNeighbors)
		for index, neighbor := range neighbors.Neighbors {
			if index > 0 {
				if neighbors.Neighbors[index-1] >= neighbor {
					return nil, nil, nil, fmt.Errorf("dag neighbors not strict increasing, neighbors: %v",
						neighbors.Neighbors)
				}
			}
			if int(neighbor) >= txIndex {
				return nil, nil, nil, fmt.Errorf("dag has neighbor >= txIndex, txIndex: %d, neighbor: %d",
					txIndex, neighbor)
			}
			dn[int(neighbor)] = struct{}{}
			if _, ok := reverseDagRemain[int(neighbor)]; !ok {
				reverseDagRemain[int(neighbor)] = make(dagNeighbors)
			}
			reverseDagRemain[int(neighbor)][txIndex] = struct{}{}
		}
		dagRemain[txIndex] = dn
	}
	return txIndexBatch, dagRemain, reverseDagRemain, nil
}

// shrinkDag shrink dag
func shrinkDag(txIndex int, dagRemain map[int]dagNeighbors,
	reverseDagRemain map[int]dagNeighbors) []int {
	var txIndexBatch []int
	for k := range reverseDagRemain[txIndex] {
		delete(dagRemain[k], txIndex)
		if len(dagRemain[k]) == 0 {
			txIndexBatch = append(txIndexBatch, k)
			delete(dagRemain, k)
		}
	}
	delete(reverseDagRemain, txIndex)
	return txIndexBatch
}
