// Copyright (C) BABEC. All rights reserved.
// Copyright (C) THL A29 Limited, a Tencent company. All rights reserved.
//
// SPDX-License-Identifier: Apache-2.0

package write

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"time"

	"chainmaker.org/chainmaker/pb-go/v2/common"
	sdk "chainmaker.org/chainmaker/sdk-go/v2"
	"chainmaker.org/chainmaker/sdk-go/v2/utils"

	com "chainweaver.org.cn/chainweaver/did/did-sync/common"
	"chainweaver.org.cn/chainweaver/did/did-sync/db/sqlite"
	"chainweaver.org.cn/chainweaver/did/did-sync/global"
	"chainweaver.org.cn/chainweaver/did/did-sync/initialize"
)

var (
	contractName = ""
	method       = "SyncTransfer"
)

type EventsCommit struct {
	eventChan <-chan *com.BlockEvent
	errorC    chan error
}

// NewBlockHandler new BlockHandler
func NewBlockHandler(eventChan <-chan *com.BlockEvent, errorC chan error) *EventsCommit {
	return &EventsCommit{
		eventChan: eventChan,
		errorC:    errorC,
	}
}

// Start Process events in the pool
func (e *EventsCommit) Start(ctx context.Context) error {
	go e.loop(ctx)
	return nil
}

// Close event subscribe close
func (e *EventsCommit) Close(err error) {
	global.LOG.Error(err.Error())
}

func (e *EventsCommit) loop(ctx context.Context) {

	state := sqlite.State{}
	for {
		select {
		case event, ok := <-e.eventChan:
			if !ok {
				global.LOG.Error("chan is close!")
				return
			}

			global.LOG.Debug(fmt.Sprintf("events %+v", event))

			err := invokeUserContract(initialize.SubSdkClient.SdkClient, true, event)
			if err != nil {
				global.LOG.Error(fmt.Sprintf("commit block[%d] events to sunchain err[%s]", event.Height, err))
				e.errorC <- fmt.Errorf("commit block[%d] events to sunchain err[%s]", event.Height, err)
				return
			}

			// block height update of sync event
			err = state.UpdateStateHeight(global.CONFIG.SubCmSdk.ChainConfig.ChainId, event.Height)
			if err != nil {
				global.LOG.Error(fmt.Sprintf("commit block[%d] events to sunchain successful,"+
					"but update databse err[%s]", event.Height, err))
			}

		case <-ctx.Done():
			return
		}
	}
}

func invokeUserContract(client *sdk.ChainClient, withSyncResult bool, events *com.BlockEvent) error {
	contractName = global.CONFIG.SubCmSdk.ContractName
	var eventList []*common.ContractEvent
	for _, list := range events.Events {
		eventList = append(eventList, list...)
	}

	eventsByte, err := json.Marshal(eventList)
	if err != nil {
		return err
	}

	// 更新合约状态数据 已同步区块高度
	str := strconv.FormatUint(events.Height, 10)

	kvs := []*common.KeyValuePair{
		{
			Key:   "eventData",
			Value: eventsByte,
		},
		{
			Key:   "blockHeight",
			Value: []byte(str),
		},
	}

	//todo circulation
	circularInvokeContract(client, contractName, method, kvs, -1, withSyncResult)
	return nil
}

func circularInvokeContract(client *sdk.ChainClient, contractName, method string, kvs []*common.KeyValuePair, timeout int64,
	withSyncResult bool) {
	for {
		txId := utils.GetTimestampTxId()
		global.LOG.Debugf("invoke contract, [contractName:%s]/[method:%s]/[txId:%s]/[kvs:%v]",
			contractName, method, txId, kvs)
		resp, err := client.InvokeContract(contractName, method, txId, kvs, timeout, withSyncResult)
		if err != nil {
			global.LOG.Errorf("The Client failed to invoke the contract: %s", err)
			time.Sleep(5 * time.Minute)
			continue
		}
		if resp.Code != common.TxStatusCode_SUCCESS {
			global.LOG.Errorf("invoke contract with [kvs:%v] failed, [code:%d]/[msg:%s]/[contractResult:%s]",
				kvs, resp.Code, resp.Message, resp.ContractResult)
			time.Sleep(5 * time.Minute)
			continue
		}

		if !withSyncResult {
			global.LOG.Infof("invoke contract success, resp: [code:%d]/[msg:%s]/[txId:%s]",
				resp.Code, resp.Message, txId)
		} else {
			global.LOG.Infof("invoke contract success, resp: [code:%d]/[msg:%s]/[contractResult:%s]",
				resp.Code, resp.Message, resp.ContractResult)
		}
		return
	}
}
