// Copyright (C) BABEC. All rights reserved.
// Copyright (C) THL A29 Limited, a Tencent company. All rights reserved.
//
// SPDX-License-Identifier: Apache-2.0

package main

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"chainweaver.org.cn/chainweaver/did/did-sync/common"
	"chainweaver.org.cn/chainweaver/did/did-sync/core"
	"chainweaver.org.cn/chainweaver/did/did-sync/global"
	"chainweaver.org.cn/chainweaver/did/did-sync/initialize"
)

func main() {
	global.GVA_VP = common.Viper() // 初始化Viper
	global.LOG = initialize.Zap(global.CONFIG.Zap)
	global.GVA_DB = initialize.Gorm() // gorm连接数据库
	if global.GVA_DB != nil {
		initialize.RegisterTables(global.GVA_DB) // 初始化表
		// 程序结束前关闭数据库链接
		db, _ := global.GVA_DB.DB()
		// 初始化db
		common.InitDbStruct(global.GVA_DB)
		defer db.Close()
	}

	//启动 main_sdk_client
	initialize.InitSdk(global.CONFIG.MainCmSdk, false, global.LOG)

	//启动 sub_sdk_client
	initialize.InitSdk(global.CONFIG.SubCmSdk, true, global.LOG)

	// new an error channel to receive errors
	errorC := make(chan error, 1)
	ctx, cancel := context.WithCancel(context.Background())

	// TODO loop listening other sync service

	err, sync := core.NewSyncHandle(ctx, errorC)
	if err != nil {
		global.LOG.Error(fmt.Sprintf("sync service encounters error: %s ", err.Error()))
		return
	}

	sync.StartSync()
	// handle exit signal in separate go routines
	go handleExitSignal(errorC)

	//global.LOG.Info(GetLogo())
	global.LOG.Info("Starting did sync service")

	err = <-errorC
	if err != nil {
		global.LOG.Error(fmt.Sprintf("sync service encounters error: %s ", err.Error()))
	}

	cancel()
	sync.Close()
	global.LOG.Info("close sync service")
}

func handleExitSignal(exitC chan<- error) {

	signalChan := make(chan os.Signal, 1)
	signal.Notify(signalChan, syscall.SIGTERM, os.Interrupt, syscall.SIGINT)
	defer signal.Stop(signalChan)

	for sig := range signalChan {
		global.LOG.Info(fmt.Sprintf("received signal: %d (%s)", sig, sig))
		exitC <- nil
	}
}

//func init() {
//	StartTime = time.Now().Format(time.RFC3339)
//}
//
//var (
//	GitCommit      string
//	GitBranch      string
//	BuildTime      string
//	StartTime      string
//	CurrentVersion string
//)
//
//const (
//	Name = "DID Sync Service"
//)
//
//func GetLogo() string {
//	// http://patorjk.com/software/taag/
//	logoName := `
//  _____ _____ _____         _____
// |  __ \_   _|  __ \       / ____|
// | |  | || | | |  | |_____| (___  _   _ _ __   ___
// | |  | || | | |  | |______\___ \| | | | '_ \ / __|
// | |__| || |_| |__| |      ____) | |_| | | | | (__
// |_____/_____|_____/      |_____/ \__, |_| |_|\___|
//                                   __/ |
//                                  |___/
//`
//	return logo.GetLogo(logoName, CurrentVersion, BuildTime, GitBranch, GitCommit, StartTime, Name)
//}
