package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"syscall"

	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/config"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/crypto"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/minio_access"
	"github.com/common-nighthawk/go-figure"
	"github.com/gin-gonic/gin"
)

var configPath = flag.String("config", "./config.yaml", "config file path")
var printVersion = flag.Bool("version", false, "print version")

func main() {
	flag.Parse()

	if *printVersion {
		PrintVersion()
		return
	}

	// init config
	if err := config.InitConfig(*configPath); err != nil {
		log.Panicf("failed to initialize config: %v", err)
		return
	}

	// init logger
	logger.Init(&config.Conf.Log)
	//logger.Infof("parse config success! config: %+v\n", config.Conf)
	logger.Info(logo())

	// init crypto
	var err error
	crypto.Csp, err = crypto.InitCrypto(&config.Conf.Crypto)
	if err != nil {
		logger.Infof("failed to initialize csp module: %v", err)
		return
	}
	logger.Info("init crypto success!")

	//init db
	err = dbaccess.InitDBAccessService(&config.Conf.DB)
	if err != nil {
		logger.Errorf("failed to initialize db module: %v", err)
		return
	}
	logger.Info("init db success!")

	//init minio
	err = minio_access.InitMinioAccessService(&config.Conf.MinioConfig)
	if err != nil {
		logger.Errorf("failed to initialize minio module: %v", err)
		return
	}
	logger.Info("init minio success!")

	// init mira-ida-access-service client
	err = service.InitMiraIdaAccessService()
	if err != nil {
		logger.Errorf("failed to initialize mira-ida-access-service client: %v", err)
		return
	}
	logger.Info("init mira-ida-access-service client success!")

	// init key service database
	err = service.InitKeyServiceAccessService()
	if err != nil {
		logger.Errorf("failed to initialize key service database: %v", err)
		return
	}
	logger.Info("init key service database success!")

	// init chainweaver.org.cn/chainweaver/mira/mira-backend-service/ service
	go func() {
		// start server
		r := gin.Default()
		r.Use(gin.Logger())
		r.Use(gin.Recovery())

		//register handlers
		ctrl.InitRouter(r)

		addr := fmt.Sprintf(":%d", config.Conf.Common.Port)
		if err := r.Run(addr); err != nil {
			panic(err)
		}
	}()
	logger.Info("Startup server success!")

	// wait for signal
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGHUP, syscall.SIGINT, syscall.SIGTERM,
		syscall.SIGQUIT, syscall.SIGUSR1, syscall.SIGUSR2, syscall.SIGTSTP)
	select {
	case sig := <-sigs:
		logger.Infof("recv signals: %d(%s)", sig, sig)
		logger.Info("Shutdown server success!")
		fmt.Fprintln(os.Stdout, "Shutdown server success!")
		os.Exit(0)
	}
}

func logo() string {
	fig := figure.NewFigure("mira-backend-service", "slant", true)
	s := fig.String()
	fragment := "================================================================================="
	versionInfo := fmt.Sprintf("Version: %2s%s\n", "", config.CurrentVersion)
	versionInfo += fmt.Sprintf("Build Time:%6s%s\n", " ", config.BuildDateTime)
	versionInfo += fmt.Sprintf("Git Branch:%6s%s\n", " ", config.GitBranch)
	versionInfo += fmt.Sprintf("Git Commit:%6s%s", " ", config.GitCommit)

	return fmt.Sprintf("\n%s\n%s%s\n%s\n", fragment, s, fragment, versionInfo)
}

func PrintVersion() {
	fmt.Println(logo())
	fmt.Println()
}
