package computingResource

import (
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/utils"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	uts "chainweaver.org.cn/chainweaver/mira/mira-common/utils"
	"encoding/json"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
	"time"
)

type CreateComputingResourceGroupHandler struct {
	req  CreateComputingResourceGroupReq
	resp common.Response
}

type CreateComputingResourceGroupReq struct {
	common.BaseRequest
	types.ComputingResourceGroup
}

func (h *CreateComputingResourceGroupHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *CreateComputingResourceGroupHandler) Process() {
	// 获取平台信息
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(h.req.ChainInfoId), h.req.RequestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
		return
	}

	// 平台ID
	partyId := cast.ToString(platformInfo.PlatformId)
	now := time.Now()
	keyId := now.Format("20060102150405")
	h.req.PartyId = partyId
	h.req.GroupId = keyId
	h.req.CreateTime = uts.TimeFormat(now)
	h.req.UpdateTime = uts.TimeFormat(now)
	h.req.Status = int32(state.ComputingResourceStatus_PUBLISHED)

	for _, n := range h.req.ComputingResourceNodeVOs {
		n.GroupId = keyId
		n.CreateTime = uts.TimeFormat(now)
		n.UpdateTime = uts.TimeFormat(now)
	}

	value, err := json.Marshal(h.req.ComputingResourceGroup)
	if err != nil {
		logger.Errorf("json marshal failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceChainFailed, err.Error())
		return
	}

	kvPairs := []*chainpb.KeyValuePair{{
		Key:   comm.ParamComputingResource,
		Value: value,
	}}

	resp, err := service.MiraIdaAccessServiceImpl.InvokeContract(h.req.RequestId, h.req.ChainInfoId, utils.ContractResource, "CreateComputingResource", kvPairs)
	if err != nil {
		logger.Errorf("create computing resource group failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceChainFailed, err.Error())
		return
	}
	h.resp.SetData(resp)
}

func (h *CreateComputingResourceGroupHandler) GetResponse() *common.Response {
	return &h.resp
}

func CreateComputingResourceGroupHandleFunc(c *gin.Context) {
	handler.Run(&CreateComputingResourceGroupHandler{}, c)
	return
}
