package computingResource

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/apis/configuration"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"github.com/gin-gonic/gin"
)

type GetComputingResourceNodeCardListPageHandler struct {
	req  configuration.GetComputingResourceNodeListPageReq
	resp common.Response
}

type GetComputingResourceNodeCardListPageReq struct {
	comm.ListInterfaceReq
}

type GetComputingResourceNodeCardListPageResp struct {
	comm.ListInterfaceResp[*types.ComputingResourceNodeVo]
}

func (h *GetComputingResourceNodeCardListPageHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetComputingResourceNodeCardListPageHandler) Process() {
	offset := h.req.PageSize * (h.req.PageNumber - 1)
	nodes, err := dbaccess.DBAccessService.GetComputingResourceNodeLocalList("", offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get computing resource node list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceListNotFound, err.Error())
		return
	}

	if nodes == nil {
		logger.Error("computing resource node list is nil")
		h.resp.SetError(common.ErrCodeDataComputingResourceListNotFound, "computing resource node list is nil")
		return
	}

	for _, node := range nodes {
		nodeId := node.NodeId
		cards, err := dbaccess.DBAccessService.GetComputingResourceCardLocalListByNodeId(nodeId)
		if err != nil {
			logger.Errorf("get computing resource card list failed, err: %v", err)
			h.resp.SetError(common.ErrCodeDataComputingResourceListNotFound, err.Error())
			return
		}
		node.ComputingResourceCardVOs = cards
	}

	count, err := dbaccess.DBAccessService.GetComputingResourceNodeLocalListCount("")
	if err != nil {
		logger.Errorf("get total of computing resource node list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceListNotFound, err.Error())
		return
	}

	resp := &GetComputingResourceNodeCardListPageResp{
		ListInterfaceResp: comm.ListInterfaceResp[*types.ComputingResourceNodeVo]{
			Pagination: comm.Pagination{
				Total:      int(count),
				PageSize:   h.req.PageSize,
				PageNumber: h.req.PageNumber,
			},
			List: nodes,
		},
	}
	h.resp.SetData(resp)
}

func (h *GetComputingResourceNodeCardListPageHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetComputingResourceNodeCardListPageHandleFunc(c *gin.Context) {
	handler.Run(&GetComputingResourceNodeCardListPageHandler{}, c)
	return
}
