package computingResource

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"github.com/gin-gonic/gin"
)

type GetGlobalComputingResourceGroupPageHandler struct {
	req  GetGlobalComputingResourceGroupPageReq
	resp common.Response
}

type GetGlobalComputingResourceGroupPageReq struct {
	common.BaseRequest
	comm.ListInterfaceReq
}

type GetGlobalComputingResourceGroupResp struct {
	types.ComputingResourceGroup
}

func (h *GetGlobalComputingResourceGroupPageHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetGlobalComputingResourceGroupPageHandler) Process() {
	query := make(map[string]interface{})
	query["chain_info_id = ?"] = h.req.ChainInfoId
	query["status = ?"] = state.ComputingResourceStatus_PUBLISHED

	offset := h.req.PageSize * (h.req.PageNumber - 1)
	value, err := dbaccess.DBAccessService.GetComputingResourceGroupList(query, offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get global computing resource group failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceChainFailed, err.Error())
		return
	}

	count, err := dbaccess.DBAccessService.GetComputingResourceGroupListCount(query)
	if err != nil {
		logger.Errorf("get total of computing resource group list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceListNotFound, err.Error())
		return
	}

	resp := &GetLocalComputingResourceGroupPageResp{
		ListInterfaceResp: comm.ListInterfaceResp[*types.ComputingResourceGroup]{
			Pagination: comm.Pagination{
				Total:      int(count),
				PageSize:   h.req.PageSize,
				PageNumber: h.req.PageNumber,
			},
			List: value,
		},
	}
	h.resp.SetData(resp)
}

func (h *GetGlobalComputingResourceGroupPageHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetGlobalComputingResourceGroupHandleFunc(c *gin.Context) {
	handler.Run(&GetGlobalComputingResourceGroupPageHandler{}, c)
	return
}
