package computingResource

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
)

type GetLocalComputingResourceGroupPageHandler struct {
	req  GetLocalComputingResourceGroupPageReq
	resp common.Response
}

type GetLocalComputingResourceGroupPageReq struct {
	common.BaseRequest
	comm.ListInterfaceReq
}

type GetLocalComputingResourceGroupPageResp struct {
	comm.ListInterfaceResp[*types.ComputingResourceGroup]
}

func (h *GetLocalComputingResourceGroupPageHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetLocalComputingResourceGroupPageHandler) Process() {
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(h.req.ChainInfoId), h.req.RequestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
		return
	}
	partyId := cast.ToString(platformInfo.PlatformId)

	query := make(map[string]interface{})
	query["party_id = ?"] = partyId
	query["chain_info_id = ?"] = h.req.ChainInfoId
	query["status = ?"] = state.ComputingResourceStatus_PUBLISHED

	offset := h.req.PageSize * (h.req.PageNumber - 1)
	value, err := dbaccess.DBAccessService.GetComputingResourceGroupList(query, offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get local computing resource group failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceChainFailed, err.Error())
		return
	}

	count, err := dbaccess.DBAccessService.GetComputingResourceGroupListCount(query)
	if err != nil {
		logger.Errorf("get total of computing resource group list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDataComputingResourceListNotFound, err.Error())
		return
	}

	resp := &GetLocalComputingResourceGroupPageResp{
		ListInterfaceResp: comm.ListInterfaceResp[*types.ComputingResourceGroup]{
			Pagination: comm.Pagination{
				Total:      int(count),
				PageSize:   h.req.PageSize,
				PageNumber: h.req.PageNumber,
			},
			List: value,
		},
	}
	h.resp.SetData(resp)
}

func (h *GetLocalComputingResourceGroupPageHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetLocalComputingResourceGroupPageHandleFunc(c *gin.Context) {
	handler.Run(&GetLocalComputingResourceGroupPageHandler{}, c)
	return
}
