package confidentialComputingModel

import (
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/utils"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	uts "chainweaver.org.cn/chainweaver/mira/mira-common/utils"
	"encoding/json"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
	"io"
	"strconv"
	"time"
)

type CreateConfidentialComputingModelHandler struct {
	req  CreateConfidentialComputingModelHandlerReq
	resp common.Response
}

type CreateConfidentialComputingModelHandlerReq struct {
	common.BaseRequest
	types.ConfidentialComputingModel
}

func (h *CreateConfidentialComputingModelHandler) BindReq(c *gin.Context) error {
	chainInfoId, err := strconv.ParseInt(c.GetHeader("X-Chain-Info-Id"), 10, 32)
	if err != nil {
		logger.Errorf("convert chainInfoId to int failed, err: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return nil
	}

	// 获取平台信息
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(chainInfoId), c.GetHeader("X-Request-Id"))
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
		return nil
	}
	// 平台ID
	partyId := cast.ToString(platformInfo.PlatformId)

	now := time.Now()
	keyId := now.Format("20060102150405")

	modelFile, _ := c.FormFile("modelFile")
	modelFileContent, _ := modelFile.Open()
	byteModelFile, err := io.ReadAll(modelFileContent)

	sourceFile, _ := c.FormFile("sourceFile")
	sourceFileContent, _ := sourceFile.Open()
	byteSourceFile, err := io.ReadAll(sourceFileContent)

	h.req.BaseRequest = common.BaseRequest{
		RequestId:   c.GetHeader("X-Request-Id"),
		ChainInfoId: int(chainInfoId),
	}

	h.req.ConfidentialComputingModel = types.ConfidentialComputingModel{
		ModelId:           keyId,
		PartyId:           partyId,
		ChainId:           int(chainInfoId),
		ModelName:         c.PostForm("name"),
		ModelType:         c.PostForm("type"),
		ModelVersion:      c.PostForm("version"),
		Category:          c.PostForm("category"),
		Description:       c.PostForm("description"),
		ModelFile:         byteModelFile,
		ModelFileHash:     c.PostForm("modelFileHash"),
		SourceFile:        byteSourceFile,
		SourceFileHash:    c.PostForm("sourceFileHash"),
		MethodName:        c.PostForm("methodName"),
		MethodDescription: c.PostForm("methodDescription"),
		Status:            int32(state.ConfidentialConputingModelStatus_PUBLISHED),
		CreateTime:        uts.TimeFormat(now),
	}

	modelParameters := c.PostForm("modelParameters")
	mp := make([]*types.ConfidentialComputingModelParameters, 0)
	if err := json.Unmarshal([]byte(modelParameters), &mp); err != nil {
		logger.Errorf("unmarshal modelParameters failed, err: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return nil
	}

	for _, m := range mp {
		m.ModelId = keyId

	}

	returnParameters := c.PostForm("returnParameters")
	rp := make([]*types.ConfidentialComputingModelReturnParameters, 0)
	if err := json.Unmarshal([]byte(returnParameters), &rp); err != nil {
		logger.Errorf("unmarshal returnParameters failed, err: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return nil
	}

	for _, r := range rp {
		r.ModelId = keyId
	}

	h.req.ModelParameters = mp
	h.req.ReturnParameters = rp
	return nil
}

func (h *CreateConfidentialComputingModelHandler) Process() {
	value, err := json.Marshal(h.req.ConfidentialComputingModel)
	if err != nil {
		logger.Errorf("json marshal failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDatConfidentialModelChainFailed, err.Error())
		return
	}

	kvPairs := []*chainpb.KeyValuePair{{
		Key:   comm.ParamComputingModel,
		Value: value,
	}}

	resp, err := service.MiraIdaAccessServiceImpl.InvokeContract(h.req.RequestId, h.req.ChainInfoId, utils.ContractResource, "CreateComputingModel", kvPairs)
	if err != nil {
		logger.Errorf("Create confidential computing model failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDatConfidentialModelChainFailed, err.Error())
		return
	}
	h.resp.SetData(resp)
}

func (h *CreateConfidentialComputingModelHandler) GetResponse() *common.Response {
	return &h.resp
}

func CreateConfidentialComputingModelHandleFunc(c *gin.Context) {
	handler.Run(&CreateConfidentialComputingModelHandler{}, c)
	return
}
