package confidentialComputingModel

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
)

type GetLocalConfidentialComputingModelPageHandler struct {
	req  GetLocalConfidentialComputingModelPageReq
	resp common.Response
}

type GetLocalConfidentialComputingModelPageReq struct {
	common.BaseRequest
	comm.ListInterfaceReq
}

type GetLocalConfidentialComputingModelPageResp struct {
	types.ConfidentialComputingModel
}

func (h *GetLocalConfidentialComputingModelPageHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetLocalConfidentialComputingModelPageHandler) Process() {
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(h.req.ChainInfoId), h.req.RequestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
		return
	}
	partyId := cast.ToString(platformInfo.PlatformId)

	query := make(map[string]interface{})
	query["party_id = ?"] = partyId
	query["chain_info_id = ?"] = h.req.ChainInfoId
	offset := h.req.PageSize * (h.req.PageNumber - 1)
	value, err := dbaccess.DBAccessService.GetConfidentialComputingModelList(query, offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get local confidential computing model failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDatConfidentialModelFailed, err.Error())
		return
	}

	count, err := dbaccess.DBAccessService.GetConfidentialComputingModelListCount(query)
	if err != nil {
		logger.Errorf("get total of cconfidential computing model list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeDatConfidentialModelFailed, err.Error())
		return
	}

	resp := &GetGlobalConfidentialComputingModelPageResp{
		ListInterfaceResp: comm.ListInterfaceResp[*types.ConfidentialComputingModel]{
			Pagination: comm.Pagination{
				Total:      int(count),
				PageSize:   h.req.PageSize,
				PageNumber: h.req.PageNumber,
			},
			List: value,
		},
	}
	h.resp.SetData(resp)
}

func (h *GetLocalConfidentialComputingModelPageHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetLocalConfidentialComputingModelPageHandleFunc(c *gin.Context) {
	handler.Run(&GetLocalConfidentialComputingModelPageHandler{}, c)
	return
}
