package configuration

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"github.com/gin-gonic/gin"
)

type GetCertListPageHandler struct {
	req  GetCertListPageReq
	resp common.Response
}

type GetCertListPageReq struct {
	comm.ListInterfaceReq
}

type GetCertListPageResp struct {
	comm.ListInterfaceResp[*types.Cert]
}

func (h *GetCertListPageHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetCertListPageHandler) Process() {
	offset := h.req.PageSize * (h.req.PageNumber - 1)
	value, err := dbaccess.DBAccessService.GetCertList("", offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get TLS list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeLocalResourceTLSFailed, err.Error())
		return
	}

	if value == nil {
		logger.Error("TLS list is nil")
		h.resp.SetError(common.ErrCodeLocalResourceTLSFailed, "TLS list is nil")
		return
	}

	count, err := dbaccess.DBAccessService.GetCertListCount("")
	if err != nil {
		logger.Errorf("get total of TLS list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeLocalResourceTLSFailed, err.Error())
		return
	}

	resp := &GetCertListPageResp{
		ListInterfaceResp: comm.ListInterfaceResp[*types.Cert]{
			Pagination: comm.Pagination{
				Total:      int(count),
				PageSize:   h.req.PageSize,
				PageNumber: h.req.PageNumber,
			},
			List: value,
		},
	}
	h.resp.SetData(resp)
}

func (h *GetCertListPageHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetCertListPageHandleFunc(c *gin.Context) {
	handler.Run(&GetCertListPageHandler{}, c)
	return
}
