package job

import (
	"chainweaver.org.cn/chainweaver/ida/key-service/pb/keypb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/config"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"encoding/json"
	"errors"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
	"strings"
	"time"
)

type CreateJobApproveHandler struct {
	req  CreateJobApproveReq
	resp common.Response
}

type CreateJobApproveReq struct {
	common.BaseRequest
	types.JobApprove
}

func (h *CreateJobApproveHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *CreateJobApproveHandler) Process() {
	// 参数校验
	if errCode, err := h.req.check(); err != nil {
		logger.Errorf("create job approve params, err: %v", err)
		h.resp.SetError(errCode, err.Error())
		return
	}

	now := time.Now()
	// 首先从数据库查询任务信息
	job, err := dbaccess.DBAccessService.GetJob(h.req.JobId)
	if err != nil {
		logger.Errorf("get job failed, err: %v", err)
		h.resp.SetError(common.ErrCodeJobDetailParameter, err.Error())
		return
	}

	// 获取平台信息
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(h.req.ChainInfoId), h.req.RequestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
		return
	}

	// 平台ID和平台名称
	partyId := cast.ToString(platformInfo.PlatformId)
	partyName := platformInfo.PlatformName

	// 初始化审批数据结构
	jobApproveExpandInfo := &types.JobApproveExpandInfo{
		PartyId:     partyId,
		ProcessType: h.req.ProcessType,
	}

	// 处理stream任务
	if h.req.ProcessType == state.ProcessType_Streaming && h.req.Status == state.JobApproveStatus_APPROVED {
		jobApproveExpandInfo.Service = h.req.Service
		if len(job.ServiceList) == 0 {
			logger.Error("job service is nil")
			h.resp.SetError(common.ErrCodeJobDetailParameter, "job service is nil")
			return
		}

		// 设置Service 本版本只支持一个party一个service
		for _, serv := range job.ServiceList {
			if serv.PartyId == h.req.Service.PartyId {
				jobApproveExpandInfo.Service.ServiceId = serv.ServiceId
				jobApproveExpandInfo.Service.ServiceName = serv.ServiceName
				jobApproveExpandInfo.Service.PartyName = serv.PartyName
				jobApproveExpandInfo.Service.ServiceClass = serv.ServiceClass
				jobApproveExpandInfo.Service.ReferExposeEndpointList = serv.ReferExposeEndpointList

				// 设置ExposeEndpointList
				originList := serv.ExposeEndpointList
				if len(originList) < 1 || len(h.req.Service.ExposeEndpointList) < 1 {
					logger.Error("invalid expose endpoint list")
					h.resp.SetError(common.ErrCodeJobDetailParameter, "invalid expose endpoint list")
					return
				}

				// 更新ExposeEndpoint
				exposeEndpoint := h.req.Service.ExposeEndpointList[0]
				originExposeEndpoint := originList[0]
				originAddress := originExposeEndpoint.Address

				originExposeEndpoint.TLSEnabled = exposeEndpoint.TLSEnabled
				originExposeEndpoint.Description = exposeEndpoint.Description
				originExposeEndpoint.ServiceCa = exposeEndpoint.ServiceCa
				originExposeEndpoint.ServiceCert = exposeEndpoint.ServiceCert
				originExposeEndpoint.ServiceKey = exposeEndpoint.ServiceKey
				originExposeEndpoint.Protocol = exposeEndpoint.Protocol
				originExposeEndpoint.Address = exposeEndpoint.Address
				originExposeEndpoint.Path = exposeEndpoint.Path
				originExposeEndpoint.Method = exposeEndpoint.Method
				originExposeEndpoint.ServiceCertName = exposeEndpoint.ServiceCertName

				for _, value := range exposeEndpoint.ValueList {
					if value.Key == "" || value.Value == "" {
						continue
					}

					isExists := false
					for _, originValue := range originExposeEndpoint.ValueList {
						if value.Key == originValue.Key {
							originValue.Value = value.Value
							isExists = true
							break
						}
					}

					if !isExists {
						originExposeEndpoint.ValueList = append(originExposeEndpoint.ValueList, value)
					}
				}

				// 更新端口信息
				if originAddress != exposeEndpoint.Address {
					splits := strings.Split(exposeEndpoint.Address, ":")

					// splits长度一定大于0
					port := splits[len(splits)-1]
					portVo := &types.Port{
						Number:      port,
						JobId:       "J__" + job.JobId + "_0",
						TaskName:    exposeEndpoint.Name,
						PartyId:     exposeEndpoint.PartyId,
						Title:       "1",
						Description: "1",
						Status:      0,
					}
					// assignPort
					if err := updatePort(config.Conf.NetworkManagerUrl+"/ports/", portVo); err != nil {
						logger.Errorf("assginPort failed, err: %v", err)
						h.resp.SetError(common.ErrCodeUpdatePortFail, err.Error())
						return
					}

					// updatePort
					portVo.Status = 1
					if err := updatePort(config.Conf.NetworkManagerUrl+"/ports/update", portVo); err != nil {
						logger.Errorf("updatePort failed, err: %v", err)
						h.resp.SetError(common.ErrCodeUpdatePortFail, err.Error())
						return
					}
				}

				//本期一个service只支持一个endpoint TODO!
				originList[0] = originExposeEndpoint
				jobApproveExpandInfo.Service.ExposeEndpointList = originList
				break
			}
		}
	}

	// 设置JobApproveResultInfo
	jobApproveResultInfo := &types.JobApproveResultInfo{
		JobId:         h.req.JobId,
		Status:        h.req.Status,
		Description:   h.req.Description,
		ApproveTime:   now.Format(time.DateTime),
		ApproveUserId: h.req.UserId,
		PartyId:       partyId,
		PartyName:     partyName,
	}

	// 设置ApprovePartyInfo
	approvePartyInfoList := make([]*types.ApprovePartyInfo, 0)
	for _, party := range job.PartyList {
		approvePartyInfo := &types.ApprovePartyInfo{
			PartyId:   party.PartyId,
			PartyName: party.PartyName,
		}
		approvePartyInfoList = append(approvePartyInfoList, approvePartyInfo)
	}
	jobApproveResultInfo.ApprovePartyInfoList = approvePartyInfoList

	// 设置JobSummaryInfo
	jobSummaryInfo := &types.JobSummaryInfo{
		JobId:           h.req.JobId,
		JobName:         job.JobName,
		CreatePartyId:   partyId,
		CreatePartyName: job.CreatePartyName,
		ModelType:       job.ModelType,
		ProcessType:     job.ProcessType,
		PartyNum:        int32(len(job.PartyList)),
		Description:     job.Description,
		CreateTime:      job.CreateTime,
	}
	jobApproveResultInfo.JobSummaryInfo = jobSummaryInfo

	// 设置JobApproveResultInfo
	jobApproveExpandInfo.JobApproveResultInfo = jobApproveResultInfo

	// 初始化EncJobApproveExpandInfo结构
	encJobApprove := &types.EncJobApproveExpandInfo{
		JobApproveExpandInfo: *jobApproveExpandInfo,
		EncService:           nil,
	}

	// 对审批任务加密处理, 目前只加密service
	if job.VisibleType != state.VisibleType_Public && jobApproveExpandInfo.Service != nil {
		partyIdList := make([]string, 0, len(job.PartyList))
		pubKeyList := make([]string, 0, len(job.PartyList))
		for _, party := range job.PartyList {
			partyIdList = append(partyIdList, party.PartyId)
			pubKeyList = append(pubKeyList, party.PubKey)
		}
		// service使用加密结构，默认不加密
		encJobService := &types.EncJobService{
			JobService: *jobApproveExpandInfo.Service,
			Private:    nil,
			Common: &types.Common{
				IsEncrypted: false,
			},
		}

		// 调用EncWithDeKWithPkList加密Service
		errCode, err := encJobServiceWithDek(encJobService, partyIdList, pubKeyList)
		if err != nil {
			logger.Errorf("encJobServiceWithDek failed, err: %v", err)
			h.resp.SetError(errCode, err.Error())
			return
		}
		encJobApprove.EncService = encJobService
		// 置空加密信息, 具体信息使用密文在链上存储
		encJobApprove.Service = nil
	}

	jobApproveBytes, err := json.Marshal(encJobApprove)
	if err != nil {
		logger.Errorf("json marshal failed, err: %v", err)
		h.resp.SetError(common.ErrCodeCreateJobApprove, err.Error())
		return
	}
	logger.Debugf("create job approve params: %s", string(jobApproveBytes))

	// 调用CreateJobApprove方法进行上链操作
	_, err = service.MiraIdaAccessServiceImpl.CreateJobApprove(h.req.RequestId, h.req.ChainInfoId, jobApproveBytes)
	if err != nil {
		logger.Errorf("create job approve failed, err: %v", err)
		h.resp.SetError(common.ErrCodeCreateJobApproveContract, err.Error())
		return
	}

	h.resp.SetData(h.req.JobId)
}

func (h *CreateJobApproveHandler) GetResponse() *common.Response {
	return &h.resp
}

func CreateJobApproveHandleFunc(c *gin.Context) {
	handler.Run(&CreateJobApproveHandler{}, c)
	return
}

// 参数校验
func (j *CreateJobApproveReq) check() (int, error) {
	//如果req的status为rejected，那么description不能为空
	if j.Status == state.JobApproveStatus_REJECTED && j.Description == "" {
		return common.ErrCodeInvalidParameterValue, errors.New("description 不能为空")
	}

	if j.Status == state.JobApproveStatus_APPROVED && j.ProcessType == state.ProcessType_Streaming &&
		(j.Service == nil || len(j.Service.ExposeEndpointList) == 0) {
		return common.ErrCodeInvalidParameterValue, errors.New("ExposeEndpoint 不能为空")
	}

	if j.Status == state.JobApproveStatus_APPROVED && j.ProcessType == state.ProcessType_Streaming {
		for _, endpoint := range j.Service.ExposeEndpointList {
			if endpoint.Protocol == "" {
				return common.ErrCodeInvalidParameterValue, errors.New("Protocol 不能为空")
			}

			if endpoint.Address == "" {
				return common.ErrCodeInvalidParameterValue, errors.New("Address 不能为空")
			}

			if endpoint.Method == "" {
				return common.ErrCodeInvalidParameterValue, errors.New("Method 不能为空")
			}

			if endpoint.TLSEnabled {
				if endpoint.ServiceCert == "" {
					return common.ErrCodeInvalidParameterValue, errors.New("ServiceCert 不能为空")
				}

				if endpoint.ServiceKey == "" {
					return common.ErrCodeInvalidParameterValue, errors.New("ServiceKey 不能为空")
				}
			}
		}
	}
	return 0, nil
}

// 加密jobStg
func encJobServiceWithDek(serv *types.EncJobService, partyIdList, pubKeyList []string) (int, error) {
	// 单独加密每一项 private表示需要加密上链的数据
	private := &types.JobServicePrivate{
		ExposeEndpointList:      serv.ExposeEndpointList,
		ReferExposeEndpointList: serv.ReferExposeEndpointList,
	}

	plaintext, err := json.Marshal(private)
	if err != nil {
		logger.Errorf("json.Marshal failed, err: %v", err)
		return common.ErrCodePutJob, err
	}

	// 调用EncWithDeKWithPkList加密
	cipherText, kekList, err := service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList(string(plaintext), "", pubKeyList, keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)
	if err != nil {
		logger.Errorf("service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList failed, err: %v", err)
		return common.ErrCodePutJob, err
	}
	// 置空加密信息, 具体信息使用密文在链上存储
	serv.ExposeEndpointList = nil
	serv.ReferExposeEndpointList = nil

	algo := keypb.DataEnvelopAlgoType_name[int32(keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)]
	serv.Common = &types.Common{
		IsEncrypted:        true,
		CipherData:         cipherText, // 密文信息
		Algo:               algo,
		AlgoExtraParamsMap: nil,
		KekList:            kekList,
		PkList:             pubKeyList,
		PartyIdList:        partyIdList,
	}

	return 0, nil
}
func NewCreateJobHandler(baseReq common.BaseRequest, job types.Job, jobName, jobId string, chainInfoId int) *CreateJobHandler {
	return &CreateJobHandler{
		req: CreateJobReq{
			BaseRequest: baseReq,
			Job:         job,
			JobName:     jobName,
			JobId:       jobId,
			ChainInfoId: chainInfoId,
		},
	}
}
