package job

import (
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/config"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"github.com/gin-gonic/gin"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"net/http/httptest"
	"strings"
	"testing"
)

func init() {
	logConfig := config.LogConfig{
		Level:    "Error",
		Module:   "mira-backend-service",
		FilePath: "../../../../cmd/logs/mira-backend-service.log",
	}
	logger.Init(&logConfig)
}

func newCreateJobApproveHandler() *CreateJobApproveHandler {
	return &CreateJobApproveHandler{
		req:  CreateJobApproveReq{},
		resp: common.Response{},
	}
}

func TestCreateJobApproveHandler_BindReq(t *testing.T) {
	handler := newCreateJobApproveHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(createJobApproveRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)
	assert.Equal(t, "5526b91f_d3cb_4f87_be0c_eae45bf18c23", handler.req.JobId)
}

func TestCreateJobApproveHandler_Process(t *testing.T) {
	handler := newCreateJobApproveHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(createJobApproveRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)

	// mock
	ctl := gomock.NewController(t)
	defer ctl.Finish()
	mockDBAccessInterface := dbaccess.NewMockDBAccessInterface(ctl)
	mockDBAccessInterface.EXPECT().GetJob(gomock.Any()).Return(&types.Job{
		JobId: handler.req.JobId,
	}, nil)

	// 初始化DB
	dbaccess.DBAccessService = mockDBAccessInterface

	mockMiraIdaAccessServiceImpl := service.NewMockMiraIdaAccessService(ctl)
	mockMiraIdaAccessServiceImpl.EXPECT().GetPlatformInfo(gomock.Any(), gomock.Any()).Return(&chainpb.PlatformData{}, nil)
	mockMiraIdaAccessServiceImpl.EXPECT().CreateJobApprove(gomock.Any(), gomock.Any(), gomock.Any()).Return("", nil)
	service.MiraIdaAccessServiceImpl = mockMiraIdaAccessServiceImpl

	// 处理请求
	handler.Process()

	// 验证响应是否包含错误
	assert.Equal(t, common.ErrCodeOK, handler.resp.Code)
	res := handler.GetResponse().Data.(string)
	assert.Equal(t, handler.req.JobId, res)
}

var createJobApproveRequestJSON = `{
  "RequestId": "34467774-ABA1-422C-BB76-962E5952FC25",
  "UserId": "123456",
  "ChainInfoId": 1,
  "jobId": "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
  "status": 1,
  "description": "般会月接式才资完上低专毛术法而。品也状每厂结却产统解权常易把。七断放二设然结再矿选较量强安公有确。近八月再对经白近叫那气角状期科。切须民新为收步方声别百共天。",
  "processType": 0,
  "service": {
    "serviceId": "85",
    "manual": true,
    "version": "sunt ullamco cillum ipsum anim",
    "partyId": "45",
    "partyName": "济越命处这采",
    "serviceClass": "amet consequat aliqua ullamco",
    "serviceName": "数员大安向集",
    "serviceLabel": "Duis laborum",
    "serviceDescription": "ad",
    "exposeEndpointList": [
      {
        "partyId": "15",
        "partyName": "群从值率商北三",
        "serviceClass": "aliqua ea consectetur dolor",
        "name": "还团里习去",
        "tlsEnabled": false,
        "description": "本拉种圆须门学要信养身清。向求水几维把深与任育际只四儿学。保易象上干理称空率题造少满三子集它展。列为易号的深效属资维四之品技。至气多年没高置运音思美角领。",
        "serviceCa": "et",
        "serviceCert": "labore eu",
        "serviceKey": "adipisicing eu labore et",
        "serviceCertName": "所写发十化红",
        "protocol": "consectetur adipisicing eu Lorem",
        "address": "山东省铜仁市瑞丽市",
        "path": "veniam deserunt in proident reprehenderit",
        "method": "exercitation qui ea in reprehenderit",
        "valueList": [
          {
            "key": "velit deserunt",
            "value": "tempor"
          }
        ]
      }
    ],
    "referExposeEndpointList": [
      {
        "id": "47",
        "name": "后机级",
        "referServiceId": "43",
        "referEndpointName": "利林切机干须",
        "serviceCa": "Lorem",
        "serviceCert": "in mollit",
        "serviceCertName": "五按品三片目",
        "protocol": "aliqua",
        "referValueList": [
          {
            "key": "est Ut aute",
            "value": "cupidatat aliquip id",
            "referServiceId": "83"
          }
        ]
      }
    ],
    "createTime": "2019-01-13 00:14:00",
    "updateTime": "2007-02-07 13:49:20"
  }
}`
