package job

import (
	"bytes"
	"encoding/json"
	"errors"
	"net/http"
	"strings"
	"time"

	"chainweaver.org.cn/chainweaver/ida/key-service/pb/keypb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/config"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/utils"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
)

type CreateJobHandler struct {
	req  CreateJobReq
	resp common.Response
}

type CreateJobReq struct {
	common.BaseRequest
	types.Job
	JobName     string            `json:"jobName" binding:"required,min=2,max=100"` // 任务名称
	PartyList   []*types.Party    `json:"partyList" binding:"required"`
	ProcessType state.ProcessType `json:"processType" binding:"omitempty,oneof=0 1 2 3"`
	ChainInfoId int               `json:"ChainInfoId"` // 链ID,Job和BaseRequest都包含此字段,需要提取出来
	JobId       string            `json:"jobId"`       // 可选的 jobId 入参
}

func (h *CreateJobHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}

	if err := h.req.check(); err != nil {
		logger.Errorf("check req error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *CreateJobHandler) Process() {
	assetDetailList := h.req.AssetDetailList
	// 根据资产可见性设置任务可见性
	isEncrypted := false
	for _, assetDetail := range assetDetailList {
		if assetDetail.VisibleType != 2 {
			isEncrypted = true
			break
		}
	}
	req := h.req
	now := time.Now()
	// 检查 JobId 字段是否为空
	var uuid string
	if req.JobId == "" {
		// 如果为空，生成 UUID 作为 jobId
		uuid = utils.UUID()
	} else {
		// 如果不为空，直接使用 JobId 字段的值作为 uuid
		uuid = req.JobId
	}

	// 获取平台信息
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(req.ChainInfoId), req.RequestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
		return
	}

	// 平台ID
	partyId := cast.ToString(platformInfo.PlatformId)
	partyName := platformInfo.PlatformName

	jobTriggerEnable := false
	if req.JobTriggerType == 0 {
		jobTriggerEnable = true
	}

	// 初始化Job信息
	job := &types.Job{
		JobId:                uuid,
		JobName:              req.JobName,
		Description:          req.Description,
		CreateTime:           now.Format(time.DateTime),
		CreateUserId:         req.UserId,
		CreatePartyName:      partyName,
		CreatePartyId:        partyId,
		ModelType:            req.ModelType,
		ProcessType:          req.ProcessType,
		PqlText:              req.PqlText,
		TaskList:             req.TaskList,
		ServiceList:          req.ServiceList,
		ResultReceiverList:   req.ResultReceiverList,
		ProcessTimes:         req.ProcessTimes,
		ProcessExecutedTimes: 0,
		ProcessEndTime:       req.ProcessEndTime,
		ProcessInterval:      req.ProcessInterval,
		ProcessUnit:          req.ProcessUnit,
		Status:               state.JobStatus_IN_APPROVAL,
		AssetDetailList:      req.AssetDetailList,
		PartyList:            req.PartyList,
		JobTriggerType:       req.JobTriggerType,
		JobTriggerEnable:     jobTriggerEnable,
		VisibleType:          state.VisibleType_Public,
		Dag:                  req.Dag,
		Topology:             req.Topology,
	}

	// 根据pubKeyName获取pubKey信息
	taskList := make([]*types.JobTask, 0)
	for _, task := range req.TaskList {
		var outputListResult []*types.Output
		for _, output := range task.OutputList {
			logger.Info("jobId: %s, PubKeyName name: ", job.JobId, output.PubKeyName)
			if output.PubKeyName != "" {
				key, err := service.KeyServiceAccessService.GetSecretKeyByPuyKeyName(output.PubKeyName)
				if err != nil {
					logger.Error("get key from pubkey name error")
					h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
					return
				}
				output.PubKey = key.PublicKey
			}
			outputListResult = append(outputListResult, output)
		}

		task.OutputList = outputListResult
		taskList = append(taskList, task)
	}
	req.TaskList = taskList

	resultReceiverListResult := make([]*types.ResultReceiver, 0)
	for _, resultReceiver := range req.ResultReceiverList {
		if resultReceiver.PubKeyName != "" {
			key, err := service.KeyServiceAccessService.GetSecretKeyByPuyKeyName(resultReceiver.PubKeyName)
			if err != nil {
				logger.Error("get key from pubkey name error")
				h.resp.SetError(common.ErrCodeGetPlatformFail, err.Error())
				return
			}
			resultReceiver.PubKey = key.PublicKey
		}
		resultReceiverListResult = append(resultReceiverListResult, resultReceiver)
	}
	req.ResultReceiverList = resultReceiverListResult

	// 设置port
	for _, serv := range req.ServiceList {
		// 更新端口信息
		for _, exposeEndpoint := range serv.ExposeEndpointList {
			splits := strings.Split(exposeEndpoint.Address, ":")
			port := splits[len(splits)-1]
			if port == types.ExposeEndpointPort {
				continue
			}

			portVo := &types.Port{
				Number:      port,
				JobId:       "J__" + uuid + "_0",
				TaskName:    exposeEndpoint.Name,
				PartyId:     exposeEndpoint.PartyId,
				Title:       "1",
				Description: "1",
				Status:      0,
			}
			// assignPort
			if err := updatePort(config.Conf.NetworkManagerUrl+"/ports/", portVo); err != nil {
				logger.Errorf("assginPort failed, err: %v", err)
				h.resp.SetError(common.ErrCodeUpdatePortFail, err.Error())
				return
			}

			// updatePort
			portVo.Status = 1
			if err := updatePort(config.Conf.NetworkManagerUrl+"/ports/update", portVo); err != nil {
				logger.Errorf("updatePort failed, err: %v", err)
				h.resp.SetError(common.ErrCodeUpdatePortFail, err.Error())
				return
			}
		}
	}

	// 根据pqlText设置modelType
	if strings.Contains(req.PqlText, state.ModelTypeText[state.JobModelType_TEE]) {
		job.ModelType = state.JobModelType_TEE
	} else if strings.Contains(req.PqlText, state.ModelTypeText[state.JobModelType_FL]) {
		job.ModelType = state.JobModelType_FL
	} else {
		job.ModelType = state.JobModelType_FQ
	}

	// 初始化JobStg, 上链时需要使用
	stg := &types.JobStg{
		JobId:                job.JobId,
		JobName:              job.JobName,
		Description:          job.Description,
		CreateTime:           job.CreateTime,
		CreateUserId:         job.CreateUserId,
		CreatePartyName:      job.CreatePartyName,
		CreatePartyId:        job.CreatePartyId,
		ModelType:            job.ModelType,
		ProcessType:          job.ProcessType,
		PqlText:              job.PqlText,
		ProcessTimes:         job.ProcessTimes,
		ProcessEndTime:       job.ProcessEndTime,
		ProcessInterval:      job.ProcessInterval,
		ProcessUnit:          job.ProcessUnit,
		AssetsList:           job.AssetDetailList,
		PartyList:            job.PartyList,
		ProcessExecutedTimes: job.ProcessExecutedTimes,
		JobTriggerType:       job.JobTriggerType,
		JobTriggerEnable:     job.JobTriggerEnable,
		Dag:                  job.Dag,
		Topology:             job.Topology,
	}
	// Stg使用加密结构，默认不加密
	encStg := &types.EncJobStg{
		JobStg:  *stg,
		Private: nil,
		Common: &types.Common{
			IsEncrypted: false,
		},
	}

	// task使用加密结构，默认不加密
	encTaskList := make([]*types.EncJobTask, 0)
	for _, task := range req.TaskList {
		encTask := &types.EncJobTask{
			JobTask: *task,
			Private: nil,
			Common: &types.Common{
				IsEncrypted: false,
			},
		}
		encTaskList = append(encTaskList, encTask)
	}
	encTask := &types.EncJobTaskList{TaskList: encTaskList}

	// service使用加密结构，默认不加密
	encServiceList := make([]*types.EncJobService, 0)
	for _, serv := range req.ServiceList {
		encService := &types.EncJobService{
			JobService: *serv,
			Private:    nil,
			Common: &types.Common{
				IsEncrypted: false,
			},
		}
		encServiceList = append(encServiceList, encService)
	}
	encService := &types.EncJobServiceList{ServiceList: encServiceList}

	// receiver使用加密结构，默认不加密
	encReceiverList := make([]*types.EncJobReceiver, 0)
	for _, receiver := range req.ResultReceiverList {
		encReceiver := &types.EncJobReceiver{
			ResultReceiver: *receiver,
			Private:        nil,
			Common: &types.Common{
				IsEncrypted: false,
			},
		}
		encReceiverList = append(encReceiverList, encReceiver)
	}
	encReceiver := &types.EncResultReceivers{ResultReceiverList: encReceiverList}

	// party使用加密结构，默认不加密
	encPartyList := make([]*types.EncParty, 0)
	for _, party := range req.PartyList {
		encParty := &types.EncParty{
			Party:   *party,
			Private: nil,
			Common: &types.Common{
				IsEncrypted: false,
			},
		}
		encPartyList = append(encPartyList, encParty)
	}
	encParty := &types.EncPartyList{PartyList: encPartyList}

	// 初始化EncJob结构
	encJob := &types.EncJob{
		Job:                *job,
		EncJobStg:          encStg,
		EncJobTaskList:     encTask,
		EncJobServiceList:  encService,
		EncJobReceiverList: encReceiver,
		EncPartyList:       encParty,
	}

	// 对任务进行加密处理, 因为合约需要使用具体字段信息, 故分字段加密Stg, Task..
	if isEncrypted {
		encJob.VisibleType = state.VisibleType_Part
		partyIdList := make([]string, 0, len(job.PartyList))
		pubKeyList := make([]string, 0, len(job.PartyList))
		for _, party := range job.PartyList {
			partyIdList = append(partyIdList, party.PartyId)
			pubKeyList = append(pubKeyList, party.PubKey)
		}

		// 调用EncWithDeKWithPkList加密JobStg
		errCode, err := encJobStgWithDek(encJob.EncJobStg, partyIdList, pubKeyList)
		if err != nil {
			logger.Errorf("encJobStgWithDek failed, err: %v", err)
			h.resp.SetError(errCode, err.Error())
			return
		}

		// 调用EncWithDeKWithPkList加密JobTaskList
		errCode, err = encJobTaskWithDek(encJob.EncJobTaskList, partyIdList, pubKeyList)
		if err != nil {
			logger.Errorf("encJobTaskWithDek failed, err: %v", err)
			h.resp.SetError(errCode, err.Error())
			return
		}

		// 调用EncWithDeKWithPkList加密JobServiceList
		errCode, err = encJobServiceListWithDek(encJob.EncJobServiceList, partyIdList, pubKeyList)
		if err != nil {
			logger.Errorf("encJobServiceListWithDek failed, err: %v", err)
			h.resp.SetError(errCode, err.Error())
			return
		}

		// 调用EncWithDeKWithPkList加密JobReceiverList
		errCode, err = encJobReceiverWithDek(encJob.EncJobReceiverList, partyIdList, pubKeyList)
		if err != nil {
			logger.Errorf("encJobReceiverWithDek failed, err: %v", err)
			h.resp.SetError(errCode, err.Error())
			return
		}

		// 调用EncWithDeKWithPkList加密EncPartyList
		errCode, err = encJobPartyWithDek(encJob.EncPartyList, partyIdList, pubKeyList)
		if err != nil {
			logger.Errorf("encJobPartyWithDek failed, err: %v", err)
			h.resp.SetError(errCode, err.Error())
			return
		}

		// 具体加密信息不可见, 需要置空具体加密信息
		encJob.PqlText = ""
		encJob.TaskList = nil
		encJob.ServiceList = nil
		encJob.ResultReceiverList = nil
		encJob.AssetDetailList = nil
		encJob.PartyList = nil
	}

	jobBytes, err := json.Marshal(encJob)
	if err != nil {
		logger.Errorf("job marshal failed, err: %v", err)
		h.resp.SetError(common.ErrCodePutJob, err.Error())
		return
	}
	logger.Debugf("create job params: %s", string(jobBytes))

	// 调用CreateJob方法进行上链操作
	_, err = service.MiraIdaAccessServiceImpl.CreateJob(req.RequestId, req.ChainInfoId, jobBytes)
	if err != nil {
		logger.Errorf("create job failed, err: %v", err)
		h.resp.SetError(common.ErrCodePutJobContract, err.Error())
		return
	}

	h.resp.SetData(job.JobId)
}

func (h *CreateJobHandler) GetResponse() *common.Response {
	return &h.resp
}

func CreateJobHandleFunc(c *gin.Context) {
	handler.Run(&CreateJobHandler{}, c)
	return
}

// 调用其他服务, 更新端口信息
func updatePort(url string, port *types.Port) error {
	paramBytes, err := json.Marshal(port)
	if err != nil {
		return err
	}
	resp, err := http.Post(url, "application/json", bytes.NewReader(paramBytes))
	if err != nil {
		return err
	}

	defer resp.Body.Close()
	buf := new(bytes.Buffer)
	_, err = buf.ReadFrom(resp.Body)
	if err != nil {
		return err
	}

	b := buf.Bytes()
	m := map[string]interface{}{}
	if err = json.Unmarshal(b, &m); err != nil {
		logger.Errorf("updatePort json.Unmarshal failed, err: %v", err)
		return err
	}
	logger.Infof("resp: %v", m)

	code, ok := m["code"]
	if !ok {
		return errors.New("code not found")
	}

	if cast.ToInt(code) != 200 {
		return errors.New("invalid code")
	}
	return nil
}

// 调用EncWithDeKWithPkList加密加密jobStg
func encJobStgWithDek(stg *types.EncJobStg, partyIdList, pubKeyList []string) (int, error) {
	// private表示需要加密上链的数据
	private := &types.JobStgPrivate{
		PqlText:    stg.PqlText,
		AssetsList: stg.AssetsList,
		PartyList:  stg.PartyList,
	}

	plaintext, err := json.Marshal(private)
	if err != nil {
		logger.Errorf("json.Marshal failed, err: %v", err)
		return common.ErrCodePutJob, err
	}

	// 调用EncWithDeKWithPkList加密
	cipherText, kekList, err := service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList(string(plaintext), "", pubKeyList, keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)
	if err != nil {
		logger.Errorf("service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList failed, err: %v", err)
		return common.ErrCodePutJob, err
	}
	// 置空加密信息, 具体信息使用密文在链上存储
	stg.PqlText = ""
	stg.AssetsList = nil
	stg.PartyList = nil

	algo := keypb.DataEnvelopAlgoType_name[int32(keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)]
	stg.Common = &types.Common{
		IsEncrypted:        true,
		CipherData:         cipherText, // 密文信息
		Algo:               algo,
		AlgoExtraParamsMap: nil,
		KekList:            kekList,
		PkList:             pubKeyList,
		PartyIdList:        partyIdList,
	}
	return 0, nil
}

// 调用EncWithDeKWithPkList加密加密jobTask
func encJobTaskWithDek(task *types.EncJobTaskList, partyIdList, pubKeyList []string) (int, error) {
	// 单独加密每一项
	for _, jobTask := range task.TaskList {
		// private表示需要加密上链的数据
		private := &types.JobTaskPrivate{
			Module:     jobTask.Module,
			Input:      jobTask.Input,
			OutputList: jobTask.OutputList,
			PartyList:  jobTask.PartyList,
		}

		plaintext, err := json.Marshal(private)
		if err != nil {
			logger.Errorf("json.Marshal failed, err: %v", err)
			return common.ErrCodePutJob, err
		}

		// 调用EncWithDeKWithPkList加密
		cipherText, kekList, err := service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList(string(plaintext), "", pubKeyList, keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)
		if err != nil {
			logger.Errorf("service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList failed, err: %v", err)
			return common.ErrCodePutJob, err
		}
		// 置空加密信息, 具体信息使用密文在链上存储
		jobTask.Module = nil
		jobTask.Input = nil
		jobTask.OutputList = nil
		jobTask.PartyList = nil

		algo := keypb.DataEnvelopAlgoType_name[int32(keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)]
		jobTask.Common = &types.Common{
			IsEncrypted:        true,
			CipherData:         cipherText, // 密文信息
			Algo:               algo,
			AlgoExtraParamsMap: nil,
			KekList:            kekList,
			PkList:             pubKeyList,
			PartyIdList:        partyIdList,
		}
	}

	return 0, nil
}

// 调用EncWithDeKWithPkList加密加密jobServiceList
func encJobServiceListWithDek(serv *types.EncJobServiceList, partyIdList, pubKeyList []string) (int, error) {
	// 单独加密每一项
	for _, jobService := range serv.ServiceList {
		// private表示需要加密上链的数据
		private := &types.JobServicePrivate{
			ExposeEndpointList:      jobService.ExposeEndpointList,
			ReferExposeEndpointList: jobService.ReferExposeEndpointList,
		}

		plaintext, err := json.Marshal(private)
		if err != nil {
			logger.Errorf("json.Marshal failed, err: %v", err)
			return common.ErrCodePutJob, err
		}

		// 调用EncWithDeKWithPkList加密
		cipherText, kekList, err := service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList(string(plaintext), "", pubKeyList, keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)
		if err != nil {
			logger.Errorf("service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList failed, err: %v", err)
			return common.ErrCodePutJob, err
		}
		// 置空加密信息, 具体信息使用密文在链上存储
		jobService.ExposeEndpointList = nil
		jobService.ReferExposeEndpointList = nil

		algo := keypb.DataEnvelopAlgoType_name[int32(keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)]
		jobService.Common = &types.Common{
			IsEncrypted:        true,
			CipherData:         cipherText, // 密文信息
			Algo:               algo,
			AlgoExtraParamsMap: nil,
			KekList:            kekList,
			PkList:             pubKeyList,
			PartyIdList:        partyIdList,
		}
	}

	return 0, nil
}

// 调用EncWithDeKWithPkList加密加密jobReceiver
func encJobReceiverWithDek(receiver *types.EncResultReceivers, partyIdList, pubKeyList []string) (int, error) {
	// 单独加密每一项
	for _, jobReceiver := range receiver.ResultReceiverList {
		// private表示需要加密上链的数据
		private := &types.JobReceiverPrivate{
			PubKeyName: "",
			PubKey:     "",
		}

		plaintext, err := json.Marshal(private)
		if err != nil {
			logger.Errorf("json.Marshal failed, err: %v", err)
			return common.ErrCodePutJob, err
		}

		// 调用EncWithDeKWithPkList加密
		cipherText, kekList, err := service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList(string(plaintext), "", pubKeyList, keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)
		if err != nil {
			logger.Errorf("service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList failed, err: %v", err)
			return common.ErrCodePutJob, err
		}
		// 置空加密信息, 具体信息使用密文在链上存储
		jobReceiver.PubKeyName = ""
		jobReceiver.PubKey = ""

		algo := keypb.DataEnvelopAlgoType_name[int32(keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)]
		jobReceiver.Common = &types.Common{
			IsEncrypted:        true,
			CipherData:         cipherText, // 密文信息
			Algo:               algo,
			AlgoExtraParamsMap: nil,
			KekList:            kekList,
			PkList:             pubKeyList,
			PartyIdList:        partyIdList,
		}
	}

	return 0, nil
}

// 调用EncWithDeKWithPkList加密加密jobParty
func encJobPartyWithDek(party *types.EncPartyList, partyIdList, pubKeyList []string) (int, error) {
	// 单独加密每一项
	for _, jobParty := range party.PartyList {
		// private表示需要加密上链的数据
		private := &types.PartyPrivate{
			ServerInfo: jobParty.ServerInfo,
		}

		plaintext, err := json.Marshal(private)
		if err != nil {
			logger.Errorf("json.Marshal failed, err: %v", err)
			return common.ErrCodePutJob, err
		}

		// 调用EncWithDeKWithPkList加密
		cipherText, kekList, err := service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList(string(plaintext), "", pubKeyList, keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)
		if err != nil {
			logger.Errorf("service.MiraIdaAccessServiceImpl.EncWithDeKWithPkList failed, err: %v", err)
			return common.ErrCodePutJob, err
		}
		// 置空加密信息, 具体信息使用密文在链上存储
		jobParty.ServerInfo = nil

		algo := keypb.DataEnvelopAlgoType_name[int32(keypb.DataEnvelopAlgoType_SM2SM4DataEnvelope)]
		jobParty.Common = &types.Common{
			IsEncrypted:        true,
			CipherData:         cipherText, // 密文信息
			Algo:               algo,
			AlgoExtraParamsMap: nil,
			KekList:            kekList,
			PkList:             pubKeyList,
			PartyIdList:        partyIdList,
		}
	}

	return 0, nil
}
func NewCreateJobApproveHandler(baseReq common.BaseRequest, jobApprove types.JobApprove) *CreateJobApproveHandler {
	return &CreateJobApproveHandler{
		req: CreateJobApproveReq{
			BaseRequest: baseReq,
			JobApprove:  jobApprove,
		},
	}
}

// 参数校验
func (r *CreateJobReq) check() error {
	const ProcessTimesInf = 99999999

	// 创建批次任务中处理次数：如果为-1，则无次数限制(设置为ProcessTimesInf)；其他需要>=1
	if r.ProcessType == state.ProcessType_Batch {
		if r.ProcessTimes < -1 || r.ProcessTimes == 0 {
			return errors.New("非法的执行次数")
		}

		if r.ProcessTimes == -1 {
			r.ProcessTimes = ProcessTimesInf
		}
	}

	// 创建周期任务中执行频率：如果为 < 30m, 则提示周期执行频率过短，至少大于30分钟
	if r.ProcessType == state.ProcessType_Cycle {
		if r.ProcessUnit < 0 || r.ProcessInterval <= 0 {
			return errors.New("非法的执行频率")
		}

		if r.ProcessUnit == 0 && r.ProcessInterval < 30 {
			return errors.New("周期执行频率过短，至少大于30分钟")
		}
		if r.ProcessEndTime == "" {

		}
		_, err := time.Parse("2006-01-02 15:04:05", r.ProcessEndTime)
		if err != nil {
			return errors.New("processEndTime格式有问题")
		}
	}

	if r.ProcessType == state.ProcessType_Single {
		if r.ProcessTimes != 1 {
			return errors.New("单次任务的processTimes 参数非法")
		}
	}

	if r.PartyList == nil || len(r.PartyList) == 0 {
		return errors.New("参数非法 缺少partyList")
	}

	return nil
}
