package job

import (
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"github.com/gin-gonic/gin"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"net/http/httptest"
	"strings"
	"testing"
)

func newCreateJobHandler() *CreateJobHandler {
	return &CreateJobHandler{
		req:  CreateJobReq{},
		resp: common.Response{},
	}
}

func TestCreateJobHandler_BindReq(t *testing.T) {
	handler := newCreateJobHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(createJobRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)
	assert.Equal(t, "5526b91f_d3cb_4f87_be0c_eae45bf18c23", handler.req.JobId)
}

func TestCreateJobHandler_BindReqProcessTimesInf(t *testing.T) {
	handler := newCreateJobHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON
	requestJson := `{
		"jobId": "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
		"jobName": "test",
		"processTimes": -1,
		"processInterval": 10,
		"processUnit": 0,
		"processType": 1
	}`
	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(requestJson))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	require.NoError(t, err)
	require.Equal(t, "5526b91f_d3cb_4f87_be0c_eae45bf18c23", handler.req.JobId)
	require.Equal(t, int32(99999999), handler.req.ProcessTimes)
}

func TestCreateJobHandler_BindReqProcessUnit(t *testing.T) {
	handler := newCreateJobHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON
	requestJson := `{
		"jobId": "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
		"jobName": "test",
		"processTimes": -1,
		"processInterval": 10,
		"processUnit": 0,
		"processType": 2
	}`
	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(requestJson))
	err := handler.BindReq(c)

	// 验证绑定是否有错误
	require.Error(t, err)
	require.EqualError(t, err, "周期执行频率过短，至少大于30分钟")
}

func TestCreateJobHandler_Process(t *testing.T) {
	handler := newCreateJobHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(createJobRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)

	// mock
	ctl := gomock.NewController(t)
	defer ctl.Finish()
	mockDBAccessInterface := dbaccess.NewMockDBAccessInterface(ctl)
	//mockDBAccessInterface.EXPECT().GetSecretKeyByPuyKeyName(gomock.Any()).Return(&types.SecretKey{}, nil)

	// 初始化DB
	dbaccess.DBAccessService = mockDBAccessInterface

	mockMiraIdaAccessServiceImpl := service.NewMockMiraIdaAccessService(ctl)
	mockMiraIdaAccessServiceImpl.EXPECT().GetPlatformInfo(gomock.Any(), gomock.Any()).Return(&chainpb.PlatformData{}, nil)
	mockMiraIdaAccessServiceImpl.EXPECT().CreateJob(gomock.Any(), gomock.Any(), gomock.Any()).Return("", nil)
	mockMiraIdaAccessServiceImpl.EXPECT().EncWithDeKWithPkList(gomock.Any(), gomock.Any(), gomock.Any(), gomock.Any()).Return("", nil, nil).Times(4)
	service.MiraIdaAccessServiceImpl = mockMiraIdaAccessServiceImpl

	// 处理请求
	handler.Process()

	// 验证响应是否包含错误
	assert.Equal(t, common.ErrCodeOK, handler.resp.Code)
	res := handler.GetResponse().Data.(string)
	assert.Equal(t, 36, len(res))
	t.Logf("jobId: %v", res)
}

var createJobRequestJSON = `{
  "RequestId": "34467774-ABA1-422C-BB76-962E5952FC25",
  "UserId": "123456",
  "ChainInfoId": 1,
  "jobId": "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
  "description": "能具增去常京每调造进气消受他始自动。联三教北水路行矿省区史快。情学特各半广备明性北厂革石管身示参。十做多高例利才究只县其时等两人五而六。",
  "createTime": "2019-12-17 15:31:58",
  "createUserId": "41",
  "createPartyName": "公界装候却极",
  "createPartyId": "27",
  "pqlText": "adipisicing elit sint officia",
  "taskList": [
    {
      "taskName": "指信更",
      "taskLabel": "Ut aute labore",
      "taskId": "36",
      "version": "dolore in pariatur amet",
      "module": {
        "moduleName": "真解可军",
        "paramList": [
          {
            "key": "incididunt",
            "value": "amet dolor"
          }
        ]
      },
      "input": {
        "taskId": "24",
        "srcTaskId": "74",
        "srcTaskName": "圆象确龙分但",
        "inputDataDetailList": [
          {
            "dataName": "问名周学",
            "dataId": "73",
            "domainId": "43",
            "domainName": "容列管历机",
            "taskSrc": "nostrud Duis cillum aliqua Ut",
            "databaseName": "历经手业七本",
            "tableName": "听系半一易",
            "columnName": "命只因身构",
            "type": "cillum ullamco",
            "length": 31,
            "comments": "pariatur aute elit Lorem exercitation",
            "assetName": "交各风该消间",
            "role": "veniam"
          }
        ]
      },
      "partyList": [
        {
          "partyId": "83",
          "partyName": "它由广众级",
          "serverInfo": {
            "ip": "216.119.12.19",
            "port": "deserunt pariatur nostrud est aute",
            "clusterId": "85"
          },
          "status": 6,
          "timestamp": "2002-02-12 06:41:39",
          "pubKey": "nulla officia deserunt esse"
        }
      ],
      "partyId": "94",
      "groupId": "38",
      "updateTime": "2017-10-16 19:22:58",
      "createTime": "2008-03-22 17:58:01"
    }
  ],
  "resultReceiverList": [
    {
      "Id": "27",
      "partyId": "13",
      "partyName": "马果理统京",
      "isEncrypted": false,
      "pubKeyName": "科县具",
      "pubKey": "est veniam pariatur id Ut"
    }
  ],
  "processTimes": -1,
  "processExecutedTimes": 10,
  "processEndTime": "1978-09-21 11:33:38",
  "processInterval": 10,
  "processUnit": 98,
  "status": 61,
  "assetDetailList": [
    {
      "assetId": "42",
      "assetName": "强如文圆",
      "visibleType": 21
    }
  ],
  "partyList": [
    {
      "partyId": "18",
      "partyName": "上团决上",
      "serverInfo": {
        "ip": "73.128.229.9",
        "port": "cupidatat officia adipisicing",
        "clusterId": "77"
      },
      "status": 39,
      "timestamp": "1975-11-14 06:51:14",
      "pubKey": "exercitation dolor proident non"
    }
  ],
  "triggerType": 54,
  "jobTriggerEnable": true,
  "visibleType": 25,
  "dag": {
    "nodeList": [
      {
        "id": "45",
        "label": "qui ut cillum",
        "shape": "mollit",
        "image": "http://dummyimage.com/400x400",
        "findLabel": "velit exercitation aute Ut anim"
      }
    ],
    "edgeList": [
      {
        "from": "in Ut cupidatat",
        "to": "sunt",
        "label": "incididunt commodo eu ipsum"
      }
    ]
  },
  "topology": {
    "nodeList": [
      {
        "name": "素支线提得",
        "status": "dolor",
        "nodeError": true,
        "dataType": "ad reprehenderit nulla",
        "totalQueries": "sint",
        "successRate": "occaecat dolore amet fugiat in",
        "averageTime": "1994-10-12 03:30:07",
        "id": "47",
        "serviceType": "Lorem aliqua deserunt id consequat"
      }
    ],
    "edgeList": [
      {
        "source": "Excepteur mollit ex",
        "target": "laboris",
        "data": {
          "type": "minim mollit tempor adipisicing"
        }
      }
    ]
  },
  "jobName": "采称表却至月",
  "modelType": 1,
  "processType": 1
}`
