/**
 * @Author: xueyanghan
 * @File: get_asset_handler.go
 * @Version: 1.0.0
 * @Description: desc.
 * @Date: 2024/5/9 18:24
 */

package job

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"encoding/json"
	"github.com/gin-gonic/gin"
)

type GetJobApproveHandler struct {
	req  GetJobApproveReq
	resp common.Response
}

type GetJobApproveReq struct {
	common.BaseRequest

	JobId string `json:"jobId"  binding:"required"` // 任务ID
}

// GetJobApproveResp 任务审批信息
type GetJobApproveResp struct {
	JobApproveInfo *JobApproveInfo `json:"jobApproveInfo"`
}

// JobApproveInfo 任务审批详情
type JobApproveInfo struct {
	types.JobApproveInfo
	LocalStatus  state.JobApproveStatus `json:"localStatus"`  // 本参与方审批状态
	ApprovedRate float64                `json:"approvedRate"` // 审批通过率
}

func (h *GetJobApproveHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetJobApproveHandler) Process() {
	// 从数据库查询任务审批信息
	jobApprove, err := dbaccess.DBAccessService.GetJobApproveInfo(h.req.JobId)
	if err != nil {
		logger.Errorf("get job approve failed. err: %v", err)
		// 数据库查询失败, 可能数据不存在, 调用链上合约, 查询链上是否有数据
		//jobApprove, err = h.GetJobApproveFromContract(h.req.RequestId, h.req.ChainInfoId, h.req.JobId)
		//if err != nil {
		//	logger.Errorf("get job approve from chain failed. err: %v", err)
		//	h.resp.SetError(common.ErrCodeGetJobApproveContract, err.Error())
		//	return
		//}
		h.resp.SetError(common.ErrCodeGetJobApprove, err.Error())
		return
	}

	jobApproveInfo := &JobApproveInfo{
		JobApproveInfo: *jobApprove,
		LocalStatus:    0,
		ApprovedRate:   float64(jobApprove.ApprovedRate) / 100, // 审批通过率转换
	}

	// 设置任务审批状态
	jobApproveInfo.setLocalStatus(uint32(h.req.ChainInfoId), h.req.RequestId)
	resp := &GetJobApproveResp{
		JobApproveInfo: jobApproveInfo,
	}
	h.resp.SetData(resp)

}

func (h *GetJobApproveHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetJobApproveHandleFunc(c *gin.Context) {
	handler.Run(&GetJobApproveHandler{}, c)
	return
}

// GetJobApproveFromContract 从链上获取任务审批信息
func (h *GetJobApproveHandler) GetJobApproveFromContract(requestId string, chainInfoId int, jobId string) (*types.JobApproveInfo, error) {
	// 调用GetJobApprove方法获取任务审批信息
	data, err := service.MiraIdaAccessServiceImpl.GetJobApprove(requestId, chainInfoId, []byte(jobId))
	if err != nil {
		logger.Errorf("get job from contract failed, err: %v", err)
		return nil, err
	}

	approve := &types.JobApproveInfo{}
	err = json.Unmarshal(data, approve)
	if err != nil {
		logger.Errorf("json unmarshal failed, err: %v", err)
		return nil, err
	}

	return approve, nil
}
