package job

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"fmt"
	"github.com/gin-gonic/gin"
	"github.com/spf13/cast"
)

type GetJobApproveListHandler struct {
	req  GetJobApproveListReq
	resp common.Response
}

type GetJobApproveListReq struct {
	common.BaseRequest
	comm.ListInterfaceReq
}

func (h *GetJobApproveListHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	h.req.ListInterfaceReq.DefaultIfNotSet()
	return nil
}

func (h *GetJobApproveListHandler) Process() {
	exp := map[string]interface{}{}

	// 根据jobName查询jobId 获取具体查询条件
	jobNameList := h.req.getJobNameList()
	if len(jobNameList) != 0 {
		jobIdList := make([]string, 0)
		for _, jobName := range jobNameList {
			if len(jobName) == 0 {
				continue
			}
			jobExp := map[string]interface{}{
				"job_name like ?": fmt.Sprintf("%%%v%%", jobName),
			}
			jobList, err := dbaccess.DBAccessService.GetJobList(jobExp, 0, 1000)
			if err != nil {
				logger.Errorf("get job list failed, err: %v", err)
				h.resp.SetError(common.ErrCodeGetJobList, err.Error())
				return
			}

			for _, job := range jobList {
				jobIdList = append(jobIdList, job.JobId)
			}
		}
		exp["job_id in ?"] = jobIdList
	}

	offset := h.req.PageSize * (h.req.PageNumber - 1)
	total, err := dbaccess.DBAccessService.GetJobApproveInfoCount(exp)
	if err != nil {
		logger.Errorf("get job approve count failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetJobApproveList, err.Error())
		return
	}

	// 从数据库查询任务审批列表
	jobApproves, err := dbaccess.DBAccessService.GetJobApproveInfoList(exp, offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get job list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetJobApproveList, err.Error())
		return
	}

	list := make([]JobApproveInfo, 0)
	for _, jobApprove := range jobApproves {
		jobApproveInfo := JobApproveInfo{
			JobApproveInfo: *jobApprove,
			LocalStatus:    0,
			ApprovedRate:   0,
		}

		// 设置任务审批状态
		jobApproveInfo.setLocalStatus(uint32(h.req.ChainInfoId), h.req.RequestId)
		list = append(list, jobApproveInfo)
	}

	resp := &comm.ListInterfaceResp[JobApproveInfo]{
		Pagination: comm.Pagination{
			Total:      int(total),
			PageSize:   h.req.PageSize,
			PageNumber: h.req.PageNumber,
		},
		List: list,
	}

	h.resp.SetData(resp)
}

func (h *GetJobApproveListHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetJobApproveListHandleFunc(c *gin.Context) {
	handler.Run(&GetJobApproveListHandler{}, c)
	return
}

// 获取参数
func (j *GetJobApproveListReq) getJobNameList() []string {
	list := make([]string, 0)
	for _, filter := range j.Filters {
		if filter.Key == "missionName" {
			list = append(list, filter.Values...)
		}
	}
	return list
}

func (j *JobApproveInfo) setLocalStatus(chainInfoId uint32, requestId string) {
	j.LocalStatus = state.JobApproveStatus_UNAPPROVED
	// 获取平台信息
	platformInfo, _ := service.MiraIdaAccessServiceImpl.GetPlatformInfo(chainInfoId, requestId)
	for _, party := range j.ApprovePartyInfoList {
		if party.PartyId == cast.ToString(platformInfo.PlatformId) {
			j.LocalStatus = party.Status
			break
		}
	}
}
