package job

import (
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"github.com/gin-gonic/gin"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"net/http/httptest"
	"strings"
	"testing"
)

func NewGetJobApproveListHandler() *GetJobApproveListHandler {
	return &GetJobApproveListHandler{
		req:  GetJobApproveListReq{},
		resp: common.Response{},
	}
}

func TestGetJobApproveListHandler_BindReq(t *testing.T) {
	handler := NewGetJobApproveListHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(getJobApproveListRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)
	assert.Equal(t, "34467774-ABA1-422C-BB76-962E5952FC25", handler.req.RequestId)
	assert.Equal(t, 1, len(handler.req.ListInterfaceReq.Filters))
	t.Log(handler.req)
}

func TestGetJobApproveListHandler_Process(t *testing.T) {
	handler := NewGetJobApproveListHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(getJobApproveListRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)

	// mock
	ctl := gomock.NewController(t)
	defer ctl.Finish()
	mockDBAccessInterface := dbaccess.NewMockDBAccessInterface(ctl)
	mockDBAccessInterface.EXPECT().GetJobList(gomock.Any(), gomock.Any(), gomock.Any()).Return([]*types.Job{}, nil).Times(2)
	mockDBAccessInterface.EXPECT().GetJobApproveInfoCount(gomock.Any()).Return(int64(1), nil)
	mockDBAccessInterface.EXPECT().GetJobApproveInfoList(gomock.Any(), gomock.Any(), gomock.Any()).Return([]*types.JobApproveInfo{
		{
			JobId: "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
			JobSummaryInfo: &types.JobSummaryInfo{
				JobId: "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
			},
		},
	}, nil)

	// 初始化DB
	dbaccess.DBAccessService = mockDBAccessInterface

	mockMiraIdaAccessServiceImpl := service.NewMockMiraIdaAccessService(ctl)
	mockMiraIdaAccessServiceImpl.EXPECT().GetPlatformInfo(gomock.Any(), gomock.Any()).Return(&chainpb.PlatformData{}, nil)
	service.MiraIdaAccessServiceImpl = mockMiraIdaAccessServiceImpl

	// 处理请求
	handler.Process()

	// 验证响应是否包含错误
	assert.Equal(t, common.ErrCodeOK, handler.resp.Code)
	res := handler.GetResponse().Data.(*comm.ListInterfaceResp[JobApproveInfo])
	assert.Equal(t, 1, len(res.List))
	assert.Equal(t, "5526b91f_d3cb_4f87_be0c_eae45bf18c23", res.List[0].JobId)
}

var getJobApproveListRequestJSON = `{
  "RequestId": "34467774-ABA1-422C-BB76-962E5952FC25",
  "UserId": "123456",
  "ChainInfoId": 1,
  "pageSize": 10,
  "pageNumber": 1,
  "filters": [
	{
      "key": "missionName",
      "values": ["name1","name2"]
    }]
}`
