/**
 * @Author: xueyanghan
 * @File: get_asset_handler.go
 * @Version: 1.0.0
 * @Description: desc.
 * @Date: 2024/5/9 18:24
 */

package job

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"encoding/json"
	"github.com/gin-gonic/gin"
	"github.com/samber/lo"
	"github.com/spf13/cast"
	"strconv"
	"strings"
)

type GetJobHandler struct {
	req  GetJobReq
	resp common.Response
}

type GetJobReq struct {
	common.BaseRequest

	JobId string `json:"jobId"  binding:"required"` // 任务ID 不能为空
}

type GetJobResp struct {
	types.Job
	ComputingResourceDetailList []*types.ComputingResourceDetail `json:"computingResourceDetailList"`
	PartyNum                    int                              `json:"partyNum"`
	JobExecuteStatus            state.JobExecuteStatus           `json:"jobExecuteStatus"`
	ApproveStatus               state.JobApproveStatus           `json:"approveStatus"`
	ChainId                     string                           `json:"chainId"`
	ChainName                   string                           `json:"chainName"`
}

func (h *GetJobHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetJobHandler) Process() {
	// 从获取任务详情
	job, err := dbaccess.DBAccessService.GetJob(h.req.JobId)
	if err != nil {
		logger.Errorf("get job failed, err: %v", err)
		// 数据库查询失败, 可能数据不存在, 调用链上合约，查询链上是否有数据
		//job, err = h.GetJobFromContract(h.req.RequestId, h.req.ChainInfoId, h.req.JobId)
		//if err != nil {
		//	logger.Errorf("get job from chain failed, err: %v", err)
		//	h.resp.SetError(common.ErrCodeGetJobContract, err.Error())
		//	return
		//}
		h.resp.SetError(common.ErrCodeGetJob, err.Error())
		return
	}

	// TODO  如果dag为空 需要根据pqltext重新解析
	if job.Dag == nil {
		logger.Warnf("getJobDagByPql because dag is nil, jobId: %s", job.JobId)
		pqlHandler := &GetJobDagByPqlHandler{
			req: GetJobDagByPqlReq{
				BaseRequest: h.req.BaseRequest,
				PqlText:     job.PqlText,
				ProcessType: job.ProcessType,
			},
		}

		pqlHandler.Process()
		graph, ok := pqlHandler.GetResponse().Data.(*types.JobGraph)
		if !ok || graph == nil {
			logger.Error("get dag by pql failed")
			h.resp.SetError(common.ErrCodeGetDAGFail, "get dag by pql failed")
			return
		}
		job.Dag = graph.Dag
		job.Topology = graph.Topology
	}

	logger.Debugf("job: %v", job)

	// 封装返回值
	resp := &GetJobResp{
		Job:                         *job,
		ComputingResourceDetailList: nil,
		PartyNum:                    0,
		JobExecuteStatus:            0,
		ApproveStatus:               jobStatusToJobApproveStatus(job.Status), // 设置任务审批状态
		ChainName:                   job.ChainName,
		ChainId:                     strconv.Itoa(job.ChainInfoId),
	}

	// 获取平台信息
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(h.req.ChainInfoId), h.req.RequestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		h.resp.SetError(common.ErrCodeUpdatePortFail, err.Error())
		return
	}

	// 平台ID和平台名称
	partyId := cast.ToString(platformInfo.PlatformId)
	partyName := platformInfo.PlatformName

	computingResourceDetails := make([]*types.ComputingResourceDetail, 0)
	for _, task := range job.TaskList {
		module := task.Module
		// 根据module信息 特殊处理
		if module == nil || len(module.ModuleName) < 3 || strings.ToUpper(module.ModuleName)[:3] != "TEE" {
			continue
		}

		// 获取计算资源信息
		computingResource, err := dbaccess.DBAccessService.GetComputingResourceByGroupIDAndPartyID(task.GroupId, task.PartyId)
		if err != nil {
			logger.Errorf("get computing resource failed, err: %v", err)
			h.resp.SetError(common.ErrCodeGetJob, err.Error())
			return
		}
		for _, computingResourceNode := range computingResource.ComputingResourceNode {
			for _, computingResourceCard := range computingResourceNode.ComputingResourceCard {
				computingResourceDetail := &types.ComputingResourceDetail{
					PartyId:     partyId,
					PartyName:   partyName,
					GroupId:     computingResourceNode.GroupID,
					GroupName:   computingResource.GroupName,
					CardSerial:  computingResourceCard.CardSerial,
					CardModel:   computingResourceCard.CardModel,
					CardSpec:    computingResourceCard.CardSpec,
					CardVersion: computingResourceCard.CardVersion,
					NodeAddress: computingResourceNode.NodeAddress,
					NodeSpec:    computingResourceNode.NodeSpec,
					TaskId:      task.TaskId,
				}
				computingResourceDetails = append(computingResourceDetails, computingResourceDetail)
			}
		}
	}

	// 设置ComputingResourceDetailList
	resp.ComputingResourceDetailList = computingResourceDetails

	// 资产名称去重
	assetNameList := lo.Uniq(lo.FilterMap(job.AssetDetailList, func(item *types.AssetDetail, index int) (string, bool) {
		return item.AssetName, true
	}))

	assetDetailList := make([]*types.AssetDetail, 0)
	for _, assetName := range assetNameList {
		// 根据资产名称获取具体资产信息, 封装最新返回值
		assetInfo, err := service.MiraIdaAccessServiceImpl.GetAssetByEnName(h.req.RequestId, h.req.ChainInfoId, assetName)
		if err != nil {
			logger.Errorf("get asset info by en name failed, err: %v", err)
			h.resp.SetError(common.ErrCodeGetAssetByEnNameFailed, err.Error())
			return
		}

		for _, item := range assetInfo.DataInfo.ItemList {
			enNames := strings.Split(assetInfo.AssetEnName, "_")
			partyId := enNames[len(enNames)-1]
			assetDetail := &types.AssetDetail{
				AssetId:       assetInfo.AssetId,
				PartyId:       partyId,
				AssetName:     assetInfo.AssetName,
				DbName:        assetInfo.DataInfo.DbName,
				TableName:     assetInfo.DataInfo.TableName,
				ColumnName:    item.Name,
				Type:          item.DataType,
				Length:        item.DataLength,
				Comments:      item.Description,
				HolderCompany: assetInfo.HolderCompany,
			}
			assetDetailList = append(assetDetailList, assetDetail)
		}
	}

	// 设置AssetDetail
	resp.AssetDetailList = assetDetailList

	jobExecuteStatus := state.JobExecuteStatus_TERMINATED
	// 查询Instance列表
	jobInstances, err := dbaccess.DBAccessService.GetJobInstanceByJobId(h.req.JobId)
	if err != nil {
		logger.Errorf("get job instance failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetInstanceByJobIdFail, err.Error())
		return
	}
	if len(jobInstances) != 0 {
		jobExecuteStatus = jobInstanceStatusToExecutingStatus(jobInstances[len(jobInstances)-1].Status)
	}
	// 设置JobExecuteStatus
	resp.JobExecuteStatus = jobExecuteStatus

	// 更新TaskLabel
	for _, task := range job.TaskList {
		if task.TaskLabel == "" {
			task.TaskLabel = task.TaskName
		}
	}

	// 更新ServiceLabel
	for _, serv := range job.ServiceList {
		if serv.ServiceLabel == "" && len(serv.ExposeEndpointList) > 0 {
			partyName := serv.ExposeEndpointList[0].PartyName
			serv.ServiceLabel = serv.ServiceName + "(" + partyName + ")"
		}

		for _, exposeEndpoint := range serv.ExposeEndpointList {
			exposeEndpoint.Description = serv.ServiceDescription
		}
	}

	h.resp.SetData(resp)
}

func (h *GetJobHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetJobHandleFunc(c *gin.Context) {
	handler.Run(&GetJobHandler{}, c)
	return
}

// 任务审批状态
func jobStatusToJobApproveStatus(jobStatus state.JobStatus) state.JobApproveStatus {
	switch jobStatus {
	case state.JobStatus_IN_APPROVAL:
		return state.JobApproveStatus_UNAPPROVED
	case state.JobStatus_APPROVE_SUCCESS:
		return state.JobApproveStatus_APPROVED
	default:
		return state.JobApproveStatus_REJECTED
	}
}

// 实例调度状态
func jobInstanceStatusToExecutingStatus(jobInstanceStatus state.JobInstanceState) state.JobExecuteStatus {
	switch jobInstanceStatus {
	case state.JobInstanceState_CANCELED, state.JobInstanceState_FAILED, state.JobInstanceState_CLOSED:
		return state.JobExecuteStatus_TERMINATED
	case state.JobInstanceState_WAITING, state.JobInstanceState_READY,
		state.JobInstanceState_RUNNING, state.JobInstanceState_CLOSING:
		return state.JobExecuteStatus_EXECUTING
	case state.JobInstanceState_SUCCESS:
		return state.JobExecuteStatus_COMPLETED
	default:
		return state.JobExecuteStatus_TERMINATED
	}
}

// GetJobFromContract 从链上获取任务信息
func (h *GetJobHandler) GetJobFromContract(requestId string, chainInfoId int, jobId string) (*types.Job, error) {
	// 从链上获取加密格式的任务信息
	data, err := service.MiraIdaAccessServiceImpl.GetJob(requestId, chainInfoId, []byte(jobId))
	if err != nil {
		logger.Errorf("get job from contract failed, err: %v", err)
		return nil, err
	}

	// 查询平台Id
	platformInfo, err := service.MiraIdaAccessServiceImpl.GetPlatformInfo(uint32(chainInfoId), requestId)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		return nil, err
	}
	platformId := cast.ToString(platformInfo.PlatformId)

	// 反序列化数据到EncJob
	encJob := &types.EncJob{}
	err = json.Unmarshal(data, encJob)
	if err != nil {
		logger.Errorf("json unmarshal failed, err: %v", err)
		return nil, err
	}

	// 解密JobStg
	stg, err := h.decryptJobStg(platformId, encJob.EncJobStg)
	if err != nil {
		logger.Errorf("decryptJobStg failed, err: %v", err)
		return nil, err
	}

	// 解密task
	taskList, err := h.decryptJobTask(platformId, encJob.EncJobTaskList)
	if err != nil {
		logger.Errorf("decryptJobTask failed, err: %v", err)
		return nil, err
	}

	// 解密Service
	serviceList, err := h.decryptJobService(platformId, encJob.EncJobServiceList)
	if err != nil {
		logger.Errorf("decryptJobService failed, err: %v", err)
		return nil, err
	}

	// 解密Receiver
	receiverList, err := h.decryptJobReceiver(platformId, encJob.EncJobReceiverList)
	if err != nil {
		logger.Errorf("decryptJobReceiver failed, err: %v", err)
		return nil, err
	}

	// 解密Party
	partyList, err := h.decryptJobParty(platformId, encJob.EncPartyList)
	if err != nil {
		logger.Errorf("decryptJobParty failed, err: %v", err)
		return nil, err
	}

	// 返回解密后的Job数据
	job := &types.Job{
		JobId:                stg.JobId,
		JobName:              stg.JobName,
		Description:          stg.Description,
		CreateTime:           stg.CreateTime,
		CreateUserId:         stg.CreateUserId,
		CreatePartyName:      stg.CreatePartyName,
		CreatePartyId:        stg.CreatePartyId,
		ModelType:            stg.ModelType,
		ProcessType:          stg.ProcessType,
		PqlText:              stg.PqlText,
		TaskList:             taskList,
		ServiceList:          serviceList,
		ResultReceiverList:   receiverList,
		ProcessTimes:         stg.ProcessTimes,
		ProcessExecutedTimes: stg.ProcessExecutedTimes,
		ProcessEndTime:       stg.ProcessEndTime,
		ProcessInterval:      stg.ProcessInterval,
		ProcessUnit:          stg.ProcessUnit,
		Status:               encJob.Status,
		AssetDetailList:      stg.AssetsList,
		PartyList:            partyList,
		JobTriggerType:       stg.JobTriggerType,
		JobTriggerEnable:     stg.JobTriggerEnable,
		Dag:                  stg.Dag,
		Topology:             stg.Topology,
		ChainName:            stg.ChainName,
		ChainInfoId:          stg.ChainInfoId,
	}

	return job, nil
}

// 解密Stg
func (h *GetJobHandler) decryptJobStg(platformId string, encJobStg *types.EncJobStg) (*types.JobStg, error) {
	// 调用DecryptPrivateData, 直接解密JobStg
	if err := service.MiraIdaAccessServiceImpl.DecryptPrivateData(platformId, encJobStg, encJobStg.Common); err != nil {
		return nil, err
	}

	return &encJobStg.JobStg, nil
}

// 解密Task
func (h *GetJobHandler) decryptJobTask(platformId string, encJobTaskList *types.EncJobTaskList) ([]*types.JobTask, error) {
	jobTaskList := make([]*types.JobTask, 0)
	for _, encTask := range encJobTaskList.TaskList {
		// 调用DecryptPrivateData, 直接解密Task
		if err := service.MiraIdaAccessServiceImpl.DecryptPrivateData(platformId, encTask, encTask.Common); err != nil {
			return nil, err
		}
		jobTaskList = append(jobTaskList, &encTask.JobTask)
	}
	return jobTaskList, nil
}

// 解密Service
func (h *GetJobHandler) decryptJobService(platformId string, encJobServiceList *types.EncJobServiceList) ([]*types.JobService, error) {
	jobServiceList := make([]*types.JobService, 0)
	for _, encServ := range encJobServiceList.ServiceList {
		// 调用DecryptPrivateData, 直接解密Service
		if err := service.MiraIdaAccessServiceImpl.DecryptPrivateData(platformId, encServ, encServ.Common); err != nil {
			return nil, err
		}
		jobServiceList = append(jobServiceList, &encServ.JobService)
	}
	return jobServiceList, nil
}

// 解密Receiver
func (h *GetJobHandler) decryptJobReceiver(platformId string, encResultReceiver *types.EncResultReceivers) ([]*types.ResultReceiver, error) {
	jobReceiverList := make([]*types.ResultReceiver, 0)
	for _, encReceiver := range encResultReceiver.ResultReceiverList {
		// 调用DecryptPrivateData, 直接解密Receiver
		if err := service.MiraIdaAccessServiceImpl.DecryptPrivateData(platformId, encReceiver, encReceiver.Common); err != nil {
			return nil, err
		}
		jobReceiverList = append(jobReceiverList, &encReceiver.ResultReceiver)
	}
	return jobReceiverList, nil
}

// 解密party
func (h *GetJobHandler) decryptJobParty(platformId string, encPartyList *types.EncPartyList) ([]*types.Party, error) {
	partyList := make([]*types.Party, 0)
	for _, encParty := range encPartyList.PartyList {
		// 调用DecryptPrivateData, 直接解密Party
		if err := service.MiraIdaAccessServiceImpl.DecryptPrivateData(platformId, encParty, encParty.Common); err != nil {
			return nil, err
		}
		partyList = append(partyList, &encParty.Party)
	}
	return partyList, nil
}
func NewGetJobHandler(baseRequest common.BaseRequest, JobId string) *GetJobHandler {
	return &GetJobHandler{
		req: GetJobReq{
			BaseRequest: baseRequest,
			JobId:       JobId,
		},
	}
}
