package job

import (
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"encoding/json"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"gorm.io/gorm"
	"net/http"
	"testing"
)

func TestGetJobHandler_Process(t *testing.T) {
	// mock
	ctl := gomock.NewController(t)
	defer ctl.Finish()

	var (
		jobId = "5526b91f_d3cb_4f87_be0c_eae45bf18c23"
	)
	mockDBAccessInterface := dbaccess.NewMockDBAccessInterface(ctl)
	mockDBAccessInterface.EXPECT().GetJob(jobId).Return(&types.Job{
		JobId:           jobId,
		AssetDetailList: []*types.AssetDetail{{AssetName: "123456"}},
		Dag:             &types.Dag{},
		Topology:        &types.Topology{},
	}, nil)
	mockDBAccessInterface.EXPECT().GetJobInstanceByJobId(jobId).Return([]*types.JobInstance{}, nil)

	// 初始化DB
	dbaccess.DBAccessService = mockDBAccessInterface

	// mock GRPC
	mockMiraIdaAccessService := service.NewMockMiraIdaAccessService(ctl)
	mockMiraIdaAccessService.EXPECT().GetPlatformInfo(gomock.Any(), gomock.Any()).Return(&chainpb.PlatformData{}, nil)
	mockMiraIdaAccessService.EXPECT().GetAssetByEnName(gomock.Any(), gomock.Any(), gomock.Any()).Return(&types.AssetInfo{
		AssetName: "123456",
	}, nil)
	service.MiraIdaAccessServiceImpl = mockMiraIdaAccessService

	handler := &GetJobHandler{
		req: GetJobReq{
			BaseRequest: common.BaseRequest{},
			JobId:       jobId,
		},
		resp: common.Response{},
	}
	handler.Process()

	resp := handler.GetResponse()
	assert.Equal(t, http.StatusOK, resp.Code)

	data := resp.Data.(*GetJobResp)
	assert.Equal(t, jobId, data.JobId)
}

func TestGetJobHandler_Process_RecordNotFound(t *testing.T) {
	// mock
	ctl := gomock.NewController(t)
	defer ctl.Finish()

	var (
		jobId = "5526b91f_d3cb_4f87_be0c_eae45bf18c23"
	)
	mockDBAccessInterface := dbaccess.NewMockDBAccessInterface(ctl)
	mockDBAccessInterface.EXPECT().GetJob(gomock.Any()).Return(nil, gorm.ErrRecordNotFound)
	mockDBAccessInterface.EXPECT().GetJobInstanceByJobId(gomock.Any()).Return([]*types.JobInstance{}, nil)

	// 初始化DB
	dbaccess.DBAccessService = mockDBAccessInterface

	// mock GRPC
	mockMiraIdaAccessService := service.NewMockMiraIdaAccessService(ctl)
	// 从链上获取Job详情
	mockMiraIdaAccessService.EXPECT().GetJob(gomock.Any(), gomock.Any(), gomock.Any()).Return(encJobJson(), nil)
	mockMiraIdaAccessService.EXPECT().GetPlatformInfo(gomock.Any(), gomock.Any()).Return(&chainpb.PlatformData{}, nil).Times(2)
	mockMiraIdaAccessService.EXPECT().DecryptPrivateData(gomock.Any(), gomock.Any(), gomock.Any()).Return(nil).Times(5)
	service.MiraIdaAccessServiceImpl = mockMiraIdaAccessService

	handler := &GetJobHandler{
		req: GetJobReq{
			BaseRequest: common.BaseRequest{},
			JobId:       jobId,
		},
		resp: common.Response{},
	}
	handler.Process()

	resp := handler.GetResponse()
	assert.Equal(t, http.StatusOK, resp.Code)

	data := resp.Data.(*GetJobResp)
	assert.Equal(t, jobId, data.JobId)
}

func encJobJson() []byte {
	data, _ := _encJobJson()
	return data
}

func _encJobJson() ([]byte, error) {
	encJob := &types.EncJob{
		Job: types.Job{
			JobId:    "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
			Dag:      &types.Dag{},
			Topology: &types.Topology{},
		},
		EncJobStg: &types.EncJobStg{
			JobStg: types.JobStg{
				JobId:    "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
				Dag:      &types.Dag{},
				Topology: &types.Topology{},
			},
			Common: &types.Common{},
		},
		EncJobTaskList: &types.EncJobTaskList{
			TaskList: []*types.EncJobTask{
				{
					JobTask: types.JobTask{
						TaskName:  "",
						TaskLabel: "",
						TaskId:    "",
					},
					Common: &types.Common{},
				},
			},
		},
		EncJobServiceList: &types.EncJobServiceList{
			ServiceList: []*types.EncJobService{
				{
					JobService: types.JobService{
						ServiceId: "",
					},
					Common: &types.Common{},
				},
			},
		},
		EncJobReceiverList: &types.EncResultReceivers{
			ResultReceiverList: []*types.EncJobReceiver{
				{
					ResultReceiver: types.ResultReceiver{
						Id: "",
					},
					Common: &types.Common{},
				},
			},
		},
		EncPartyList: &types.EncPartyList{
			PartyList: []*types.EncParty{
				{
					Party: types.Party{
						PartyId:   "",
						PartyName: "",
						PubKey:    "",
					},
					Common: &types.Common{},
				},
			},
		},
	}

	return json.Marshal(encJob)
}
