package job

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"github.com/gin-gonic/gin"
	"strconv"
	"time"
)

type GetJobInstanceListHandler struct {
	req  GetJobInstanceListReq
	resp common.Response
}

type GetJobInstanceListReq struct {
	common.BaseRequest
	comm.ListInterfaceReq

	JobId string `json:"jobId"` // 任务ID
}

// JobInstanceInfo 任务实例详情
type JobInstanceInfo struct {
	JobId           string                 `json:"jobId"`           // 任务ID
	JobInstanceId   string                 `json:"jobInstanceId"`   // 任务实例ID
	JobInstanceName string                 `json:"jobInstanceName"` // 任务实例名称
	JobType         state.ModelType        `json:"jobType"`         // 任务类型 枚举值
	Status          state.JobExecuteStatus `json:"status"`          // 任务调度状态
	FailReason      string                 `json:"failReason"`      // 调度失败原因
	StartTime       string                 `json:"startTime"`       // 调度开始时间
	EndTime         string                 `json:"endTime"`         // 调度结束时间
	Parties         []string               `json:"parties"`         // 参与方
	ChainId         string                 `json:"chainId"`
	ChainName       string                 `json:"chainName"`
}

func (h *GetJobInstanceListHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	h.req.ListInterfaceReq.DefaultIfNotSet()
	return nil
}

func (h *GetJobInstanceListHandler) Process() {

	// 查询任务实例数量
	total, err := dbaccess.DBAccessService.GetJobInstanceCountByJobId(h.req.JobId)
	if err != nil {
		logger.Errorf("get job instance count failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetInstanceByJobIdFail, err.Error())
		return
	}

	offset := h.req.PageSize * (h.req.PageNumber - 1)
	// 分页查询任务实例信息
	jobInstances, err := dbaccess.DBAccessService.GetJobInstanceListLimitByJobId(h.req.JobId, offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get job instance list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetInstanceByJobIdFail, err.Error())
		return
	}

	list := make([]JobInstanceInfo, 0)
	for _, jobInstance := range jobInstances {
		item := JobInstanceInfo{
			JobId:           h.req.JobId,
			JobInstanceId:   jobInstance.JobInstanceId,
			JobInstanceName: jobInstance.JobInstanceName,
			JobType:         jobInstance.JobInstanceType,
			Status:          jobInstanceStatusToExecutingStatus(jobInstance.Status),
			FailReason:      jobInstance.FailedReason,
			StartTime:       jobInstance.CreateTime,
			EndTime:         "", // TODO
			Parties:         jobInstance.PartyList,
			ChainName:       jobInstance.ChainName,
			ChainId:         strconv.Itoa(jobInstance.ChainInfoId),
		}

		// 解析实例更新时间
		if _, err := time.ParseInLocation(time.DateTime, jobInstance.UpdateTime, time.Local); err == nil {
			item.EndTime = jobInstance.UpdateTime
		}

		list = append(list, item)
	}

	resp := &comm.ListInterfaceResp[JobInstanceInfo]{
		Pagination: comm.Pagination{
			Total:      int(total),
			PageSize:   h.req.PageSize,
			PageNumber: h.req.PageNumber,
		},
		List: list,
	}

	h.resp.SetData(resp)
}

func (h *GetJobInstanceListHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetJobInstanceListHandleFunc(c *gin.Context) {
	handler.Run(&GetJobInstanceListHandler{}, c)
	return
}
