package job

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"fmt"
	"github.com/gin-gonic/gin"
	"strconv"
)

type GetJobListHandler struct {
	req  GetJobListReq
	resp common.Response
}

type GetJobListReq struct {
	common.BaseRequest
	comm.ListInterfaceReq
}

func (h *GetJobListHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	h.req.ListInterfaceReq.DefaultIfNotSet()
	return nil
}

func (h *GetJobListHandler) Process() {
	// 数据库查询条件
	exp := map[string]interface{}{}
	if h.req.getJobName() != "" {
		exp["job_name like ?"] = fmt.Sprintf("%%%s%%", h.req.getJobName())
	}
	if h.req.getApproveStatus() != -1 {
		exp["status = ?"] = h.req.getApproveStatus()
	}

	offset := h.req.PageSize * (h.req.PageNumber - 1)
	total, err := dbaccess.DBAccessService.GetJobCount(exp)
	if err != nil {
		logger.Errorf("get job list count, err: %v", err)
		h.resp.SetError(common.ErrCodeGetJobList, err.Error())
		return
	}

	jobs, err := dbaccess.DBAccessService.GetJobList(exp, offset, h.req.PageSize)
	if err != nil {
		logger.Errorf("get job list failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetJobList, err.Error())
		return
	}

	list := make([]types.JobInfo, 0)
	for _, job := range jobs {
		// 设置任务审批状态
		approveStatus := jobStatusToJobApproveStatus(job.Status)
		jobExecuteStatus := state.JobExecuteStatus_TERMINATED
		// 查询Instance信息
		jobInstances, err := dbaccess.DBAccessService.GetJobInstanceByJobId(job.JobId)
		if err != nil {
			logger.Errorf("get job instance by jobId failed, err: %v", err)
			h.resp.SetError(common.ErrCodeGetInstanceByJobIdFail, err.Error())
			return
		}
		if len(jobInstances) != 0 {
			// 设置任务调度状态
			jobExecuteStatus = jobInstanceStatusToExecutingStatus(jobInstances[len(jobInstances)-1].Status)
		}
		item := types.JobInfo{
			JobId:            job.JobId,
			JobName:          job.JobName,
			Description:      job.Description,
			PartyName:        job.CreatePartyName,
			CreatePartyId:    job.CreatePartyId,
			CreateTime:       job.CreateTime,
			ModelType:        job.ModelType,
			ProcessType:      job.ProcessType,
			Status:           job.Status,
			JobExecuteStatus: jobExecuteStatus,
			ApproveStatus:    approveStatus,
			ChainId:          strconv.Itoa(job.ChainInfoId),
			ChainName:        job.ChainName,
			JobTriggerEnable: job.JobTriggerEnable,
		}
		list = append(list, item)
	}

	resp := &comm.ListInterfaceResp[types.JobInfo]{
		Pagination: comm.Pagination{
			Total:      int(total),
			PageSize:   h.req.PageSize,
			PageNumber: h.req.PageNumber,
		},
		List: list,
	}

	h.resp.SetData(resp)
}

func (h *GetJobListHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetJobListHandleFunc(c *gin.Context) {
	handler.Run(&GetJobListHandler{}, c)
	return
}

// 获取JobName
func (r *GetJobListReq) getJobName() string {
	for _, filter := range r.ListInterfaceReq.Filters {
		if "name" == filter.Key && len(filter.Values) > 0 {
			return filter.Values[0]
		}
	}
	return ""
}

// 获取任务审批状态
func (r *GetJobListReq) getApproveStatus() int {
	for _, filter := range r.ListInterfaceReq.Filters {
		if "approveStatus" == filter.Key && len(filter.Values) > 0 {
			status, _ := strconv.Atoi(filter.Values[0])
			return status
		}
	}
	return -1
}
