package job

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"github.com/gin-gonic/gin"
	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/assert"
	"net/http/httptest"
	"reflect"
	"strings"
	"testing"
)

func NewGetJobListHandler() *GetJobListHandler {
	return &GetJobListHandler{
		req:  GetJobListReq{},
		resp: common.Response{},
	}
}

func TestGetJobListHandler_BindReq(t *testing.T) {
	handler := NewGetJobListHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(getJobListRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)
	assert.Equal(t, "34467774-ABA1-422C-BB76-962E5952FC25", handler.req.RequestId)
	assert.Equal(t, 2, len(handler.req.ListInterfaceReq.Filters))
	assert.Equal(t, "name", handler.req.ListInterfaceReq.Filters[0].Key)
	assert.Equal(t, "approveStatus", handler.req.ListInterfaceReq.Filters[1].Key)
	assert.Equal(t, true, reflect.DeepEqual([]string{"name1", "name2"}, handler.req.ListInterfaceReq.Filters[0].Values))
	assert.Equal(t, true, reflect.DeepEqual([]string{"2", "1"}, handler.req.ListInterfaceReq.Filters[1].Values))
	t.Log(handler.req)
}

func TestGetJobListHandler_Process(t *testing.T) {
	handler := NewGetJobListHandler()
	// 准备 HTTP 请求上下文
	c, _ := gin.CreateTestContext(httptest.NewRecorder())

	// 准备请求 JSON

	// 绑定请求
	c.Request = httptest.NewRequest("POST", "/test", strings.NewReader(getJobListRequestJSON))
	err := handler.BindReq(c)

	// 验证绑定是否成功
	assert.NoError(t, err)

	// mock
	ctl := gomock.NewController(t)
	defer ctl.Finish()
	mockDBAccessInterface := dbaccess.NewMockDBAccessInterface(ctl)
	mockDBAccessInterface.EXPECT().GetJobCount(gomock.Any()).Return(int64(1), nil)
	mockDBAccessInterface.EXPECT().GetJobList(gomock.Any(), gomock.Any(), gomock.Any()).Return([]*types.Job{
		{
			JobId: "5526b91f_d3cb_4f87_be0c_eae45bf18c23",
		},
	}, nil)
	mockDBAccessInterface.EXPECT().GetJobInstanceByJobId(gomock.Any()).Return([]*types.JobInstance{}, nil)

	// 初始化DB
	dbaccess.DBAccessService = mockDBAccessInterface

	mockMiraIdaAccessServiceImpl := service.NewMockMiraIdaAccessService(ctl)
	//mockMiraIdaAccessServiceImpl.EXPECT().GetPlatformInfo(gomock.Any(), gomock.Any()).Return(&chainpb.PlatformData{}, nil)
	service.MiraIdaAccessServiceImpl = mockMiraIdaAccessServiceImpl

	// 处理请求
	handler.Process()

	// 验证响应是否包含错误
	assert.Equal(t, common.ErrCodeOK, handler.resp.Code)
	res := handler.GetResponse().Data.(*comm.ListInterfaceResp[types.JobInfo])
	assert.Equal(t, 1, len(res.List))
	assert.Equal(t, "5526b91f_d3cb_4f87_be0c_eae45bf18c23", res.List[0].JobId)
}

var getJobListRequestJSON = `{
  "RequestId": "34467774-ABA1-422C-BB76-962E5952FC25",
  "UserId": "123456",
  "ChainInfoId": 1,
  "pageSize": 10,
  "pageNumber": 1,
  "filters": [
	{
      "key": "name",
      "values": ["name1","name2"]
    },
	{
      "key": "approveStatus",
      "values": ["2","1"]
    }]
}`
