package job

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"time"

	"chainweaver.org.cn/chainweaver/ida/key-service/pb/keypb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/service"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/minio_access"
	"github.com/gin-gonic/gin"
)

type GetResultHandler struct {
	req  GetResultReq
	resp common.Response
}

type GetMinioResultUrlReq struct {
	ObjectName string `json:"object_name"`
}

type GetResultReq struct {
	common.BaseRequest
	JobInstanceId string `json:"jobInstanceId"  binding:"required"`
}

type GetResultResp struct {
	JobID string         `json:"jobID"`
	List  []ResultDetail `json:"list"`
}

// ResultDetail 结构体
type ResultDetail struct {
	EncURL  string `json:"encUrl"`
	DataID  string `json:"dataId"`
	PartyId string `json:"partyId"`
	DecURL  string `json:"decUrl"`
}

func (h *GetResultHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetResultHandler) Process() {
	var result GetResultResp
	jobInstanceId := h.req.JobInstanceId
	result.JobID = jobInstanceId
	jobTaskInstance, err := dbaccess.DBAccessService.GetJobInstanceTaskByJobInstanceId(jobInstanceId)
	if err != nil {
		logger.Errorf("get job task failed. err: %v", err)
		h.resp.SetError(common.ErrCodeGetJobApprove, err.Error())
		return
	}

	for _, instance := range jobTaskInstance {
		for _, output := range instance.OutputList {
			if output.IsFinalResult {
				if output.DomainId != "" {
					var decUrl string
					address, err := dbaccess.DBAccessService.GetJobInstanceResultServerAddr(h.req.ChainInfoId, output.DomainId)
					if err != nil || address == "" {
						logger.Errorf("get address by domainId failed. err: %v", err.Error())
						h.resp.SetError(common.ErrCodeGetAddressByDomainId, err.Error())
						return
					}
					// 请求下载结果
					jobID := jobInstanceId
					objectName := jobID + "/" + output.DataId
					logger.Info("request minio result:", objectName)
					encUrl, err := h.req.getMinioUrlResult(address, objectName)
					if err != nil {
						logger.Errorf("get minio enc result failed. err: %v", err)
						h.resp.SetError(common.ErrGetMinioResult, err.Error())
						return
					}
					logger.Info("get minio url is: ", encUrl)
					if output.PubKey != "" && output.PubKeyName != "" {
						logger.Infof("get PubKey is %s, task id is %s:", output.PubKey, output.DomainId)
						content, err := h.req.getURLContent(encUrl)
						if err != nil {
							logger.Errorf("get url content failed. err: %v", err)
							h.resp.SetError(common.ErrCodeGetUrlContent, err.Error())
							return
						}
						// 判断是否能正确获取明文内容
						if isMinioKeyNotCorrect(content) {
							logger.Error("minio key is not correct")
							h.resp.SetError(common.ErrCodeMinioKeyIsNotCorrect, "minio key is not correct")
							return
						}
						logger.Info("begin to decrypt with: ", content)
						//	调用key service解密
						plainText, err := service.MiraIdaAccessServiceImpl.Decrypt(content, output.PubKey, keypb.AlgoType_SM2)
						if err != nil {
							logger.Errorf("service.MiraIdaAccessServiceImpl.Decrypt failed, err: %v", err)
							h.resp.SetError(common.ErrCodeDecrypt, err.Error())
							return
						}
						logger.Info("get plain text is: ", plainText)
						if output.DataId == "" {
							logger.Error("dataId is empty")
							h.resp.SetError(common.ErrCodeDataIdIsEmpty, "dataId is empty")
							return
						}
						// 存储明文
						filePath := "./data/" + output.DataId + "_" + getTimeString()
						err = h.req.saveContentToFile(plainText, filePath)
						if err != nil {
							logger.Errorf("save content to file error: %v", err)
							h.resp.SetError(common.ErrCodeSaveContentToFile, err.Error())
							return
						}
						logger.Info("save content to file success")
						// 上传明文到minio，并获取地址
						decObjectName := jobInstanceId + "/" + "result_" + output.DataId
						// 判断结果是否已经存储
						exit, err := minio_access.MinioAccessService.ObjectExits(decObjectName)
						if err != nil {
							logger.Errorf("error minio object exited. err: %v", err)
							h.resp.SetError(common.ErrGetMinioResult, err.Error())
							return
						}
						if exit {
							// 如果已经存储，直接获取结果
							decUrl, err = minio_access.MinioAccessService.GetResultUrl(decObjectName)
							if err != nil {
								logger.Errorf("get minio dec result failed. err: %v", err)
								h.resp.SetError(common.ErrGetMinioResult, err.Error())
								return
							}
							logger.Info("get minio dec result is: ", decUrl)
						} else {
							decUrl, err = minio_access.MinioAccessService.UploadFile(decObjectName, filePath)
							if err != nil {
								logger.Errorf("upload file to minio error: %v", err)
								h.resp.SetError(common.ErrCodeUploadMinio, err.Error())
								return
							}
							logger.Info("upload file to minio success: ", decUrl)
						}
					} else {
						decUrl = encUrl
					}
					resultDetail := ResultDetail{
						EncURL:  encUrl,
						DataID:  output.DataId,
						PartyId: output.DomainId,
						DecURL:  decUrl,
					}
					result.List = append(result.List, resultDetail)
				}
			}
		}
		h.resp.SetData(result)
	}
}

func (h *GetResultHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetResultHandleFunc(c *gin.Context) {
	handler.Run(&GetResultHandler{}, c)
	return
}

func (r *GetResultReq) getURLContent(url string) (string, error) {
	resp, err := http.Get(url)
	if err != nil {
		return "", err
	}
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return "", err
	}
	return string(body), nil
}

// saveContentToFile 将内容保存到指定文件
func (r *GetResultReq) saveContentToFile(content, filePath string) error {
	dir := filepath.Dir(filePath)
	if err := os.MkdirAll(dir, 0755); err != nil {
		return fmt.Errorf("failed to create directory: %v", err)
	}
	err := ioutil.WriteFile(filePath, []byte(content), 0644)
	if err != nil {
		return err
	}
	return nil
}

// saveContentToFile 将内容保存到指定文件
func (r *GetResultReq) getMinioUrlResult(address string, objectName string) (string, error) {
	url := "http://" + address + "/v1/mira/job/GetMinioUrl"
	requestData := GetMinioResultUrlReq{
		ObjectName: objectName,
	}
	jsonData, err := json.Marshal(requestData)
	if err != nil {
		return "", err
	}
	req, err := http.NewRequest("POST", url, bytes.NewBuffer(jsonData))
	if err != nil {
		return "", err
	}
	req.Header.Set("Content-Type", "application/json")
	chainInfoIdx := fmt.Sprintf("%d", r.ChainInfoId)
	req.Header.Set("X-Chain-Info-Id", chainInfoIdx)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return "", err
	}
	defer resp.Body.Close()

	var response Response

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return "", err
	}
	err = json.Unmarshal(body, &response)
	if err != nil {
		return "", err
	}
	return response.Data.ResultUrl, nil
}

func getTimeString() string {
	currentTime := time.Now()

	unixTimestamp := currentTime.Unix()
	unixTimestampStr := fmt.Sprintf("%d", unixTimestamp)
	return unixTimestampStr
}

// isMinioKeyCorrect 判断是否能正确获取minio的内容
func isMinioKeyNotCorrect(content string) bool {
	substr := "key does not exist"
	return strings.Contains(content, substr)
}
