package jobInstance

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/dbaccess"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/state"
	"github.com/gin-gonic/gin"
)

type GetJobStatisticInfoHandler struct {
	req  GetJobStatisticInfoReq
	resp common.Response
}

type GetJobStatisticInfoReq struct {
	common.BaseRequest
	JobId string `json:"jobId"`
}

type GetJobStatisticInfoResp struct {
	ExecuteCount int `json:"executeCount"`
	TotalCount   int `json:"totalCount"`
	SuccessCount int `json:"successCount"`
	FailCount    int `json:"failCount"`
	ReadyCount   int `json:"readyCount"`
}

func (h *GetJobStatisticInfoHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h.req); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetJobStatisticInfoHandler) Process() {
	jobInstanceList, err := dbaccess.DBAccessService.GetJobInstanceByJobId(h.req.JobId)
	if err != nil {
		logger.Errorf("get job instance failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetInstanceByJobIdFail, err.Error())
		return
	}

	job, err := dbaccess.DBAccessService.GetJob(h.req.JobId)
	if err != nil {
		logger.Errorf("get job failed, err: %v", err)
		h.resp.SetError(common.ErrCodeGetJobList, err.Error())
		return
	}

	successCount := 0
	failCount := 0
	readyCount := 0
	totalCount := job.ProcessTimes
	executeCount := 0

	for _, jobInstance := range jobInstanceList {
		logger.Info("jobInstance ", jobInstance)
		executeCount += 1
		jobInstanceStatus2ExecutingStatus := JobInstanceStatus2ExecutingStatus(jobInstance.Status)
		if jobInstanceStatus2ExecutingStatus == state.JobExecuteStatus_COMPLETED {
			successCount++
		} else if jobInstanceStatus2ExecutingStatus == state.JobExecuteStatus_TERMINATED {
			failCount++
		} else {
			readyCount++
		}
	}

	resp := &GetJobStatisticInfoResp{
		ExecuteCount: executeCount,
		TotalCount:   int(totalCount),
		SuccessCount: successCount,
		FailCount:    failCount,
		ReadyCount:   readyCount,
	}
	h.resp.SetData(resp)
}

func (h *GetJobStatisticInfoHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetJobStatisticInfoHandleFunc(c *gin.Context) {
	handler.Run(&GetJobStatisticInfoHandler{}, c)
	return
}

func JobInstanceStatus2ExecutingStatus(jobInstanceStatus state.JobInstanceState) state.JobExecuteStatus {
	var jobExecuteStatus state.JobExecuteStatus
	switch jobInstanceStatus {
	case state.JobInstanceState_CANCELED, state.JobInstanceState_FAILED, state.JobInstanceState_CLOSED:
		jobExecuteStatus = state.JobExecuteStatus_TERMINATED
	case state.JobInstanceState_WAITING, state.JobInstanceState_READY, state.JobInstanceState_RUNNING, state.JobInstanceState_CLOSING:
		jobExecuteStatus = state.JobExecuteStatus_EXECUTING
	case state.JobInstanceState_SUCCESS:
		jobExecuteStatus = state.JobExecuteStatus_COMPLETED
	}
	return jobExecuteStatus
}
