package port

import (
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/config"
	"encoding/json"
	"io/ioutil"
	"net/http"

	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/common"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/ctrl/handler"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"github.com/gin-gonic/gin"
)

type GetAvailableAddressListHandler struct {
	resp common.Response
}

// AvailableListResponse 获取空闲地址列表响应结构体
type AvailableListResponse struct {
	AvailableList []string `json:"availableList"`
}

func (h *GetAvailableAddressListHandler) BindReq(c *gin.Context) error {
	if err := c.ShouldBindJSON(&h); err != nil {
		logger.Errorf("BindReq error: %v", err)
		h.resp.SetError(common.ErrCodeInvalidParameter, err.Error())
		return err
	}
	return nil
}

func (h *GetAvailableAddressListHandler) Process() {
	resp, err := GetAvailableAddressList(config.Conf.NetworkManagerUrl)
	if err != nil {
		logger.Errorf("get port server available addr error: %v", err)
		h.resp.SetError(common.ErrCodeGetAvailableAddrList, err.Error())
		return
	}
	h.resp.SetData(resp)
}

func (h *GetAvailableAddressListHandler) GetResponse() *common.Response {
	return &h.resp
}

func GetAvailableAddressListHandleFunc(c *gin.Context) {
	handler.Run(&GetAvailableAddressListHandler{}, c)
	return
}

// GetAvailableAddressList 获取空闲端口地址
func GetAvailableAddressList(portServerAddr string) (*AvailableListResponse, error) {
	url := portServerAddr + "/ports/GetAvailableAddressList"
	resp, err := http.Get(url)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode != http.StatusOK {
		return nil, err
	}
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	var genericResponse common.Response
	err = json.Unmarshal(body, &genericResponse)
	if err != nil {
		return nil, err
	}
	dataJson, err := json.Marshal(genericResponse.Data)
	if err != nil {
		return nil, err
	}
	var availableListResponse AvailableListResponse
	err = json.Unmarshal(dataJson, &availableListResponse)
	if err != nil {
		return nil, err
	}
	logger.Debugf("Available List: %v\n", availableListResponse.AvailableList)
	return &availableListResponse, nil
}
