/**
 * @Author: xueyanghan
 * @File: ida_access_service.go
 * @Version: 1.0.0
 * @Description: desc.
 * @Date: 2024/5/9 16:03
 */

package service

import (
	"bytes"
	"chainweaver.org.cn/chainweaver/ida/chain-service/pb/chainpb"
	"chainweaver.org.cn/chainweaver/ida/common/grpc"
	"chainweaver.org.cn/chainweaver/ida/key-service/pb/keypb"
	"chainweaver.org.cn/chainweaver/ida/registration-service/pb/registrationpb"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/config"
	"chainweaver.org.cn/chainweaver/mira/mira-backend-service/core/utils"
	"chainweaver.org.cn/chainweaver/mira/mira-common/logger"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types"
	"chainweaver.org.cn/chainweaver/mira/mira-common/types/comm"
	"chainweaver.org.cn/chainweaver/mira/mira-ida-access-service/pb"
	"context"
	"encoding/binary"
	"encoding/json"
	"fmt"
	"github.com/pkg/errors"
	"github.com/zeromicro/go-zero/zrpc"
	"net/http"
	"strconv"
	"time"
)

// MiraIdaAccessService 接口定义了与ida-access交互的GRPC方法。
type MiraIdaAccessService interface {
	// GetAssetList 获取资产列表
	GetAssetList(requestId string, chainInfoId int, req comm.ListInterfaceReq) (*comm.Pagination, []types.AssetItem, error)
	// GetAssetInfo 获取指定资产的信息
	GetAssetInfo(requestId string, chainInfoId int, assetId string) (*types.AssetInfo, error)
	// GetAssetByEnName 通过资产英文名获取资产信息
	GetAssetByEnName(requestId string, chainInfoId int, assetEnName string) (*types.AssetInfo, error)
	// GetEnterpriseList 获取企业列表
	GetEnterpriseList(requestId string, chainInfoId int, assetName string) (*types.AssetEnterpriseList, error)
	// CreateJob 任务上链
	CreateJob(requestId string, chainInfoId int, params []byte) (string, error)
	// CreateJobApprove 任务审批上链
	CreateJobApprove(requestId string, chainInfoId int, params []byte) (string, error)
	// TriggerJobEnable 触发任务
	TriggerJobEnable(requestId string, chainInfoId int, jobId string) (string, error)
	// CancelJobInstance 取消任务实例
	CancelJobInstance(requestId string, chainInfoId int, jobInstanceId string) (string, error)
	// GetPlatformInfo 获取平台信息
	GetPlatformInfo(chainInfoId uint32, requestId string) (*chainpb.PlatformData, error)
	GetListChainInfo(requestId string) (*chainpb.ListChainInfoData, error)
	// EncWithDeK 数字信封加密 使用公钥和额外参数加密明文，并返回密文和密钥
	EncWithDeK(plainText, pubKey, extraParam string, dataEnvelopAlgoType keypb.DataEnvelopAlgoType) (string, string, error)
	// DecByKeK 数字信封解密 使用密钥和公钥解密密文，返回明文
	DecByKeK(cipherText, pubKey, kek string, dataEnvelopAlgoType keypb.DataEnvelopAlgoType) (string, error)
	// Encrypt 使用公钥和额外参数加密明文，返回密文
	Encrypt(plainText, pubKey, extraParam string, algoType keypb.AlgoType) (string, error)
	// Decrypt 使用公钥解密密文，返回明文
	Decrypt(cipherText, pubKey string, algoType keypb.AlgoType) (string, error)
	// EncWithDeKWithPkList 使用多个公钥和额外参数加密明文，返回密文和密钥列表
	EncWithDeKWithPkList(plainText, extraParam string, pubKeyList []string, dataEnvelopAlgoType keypb.DataEnvelopAlgoType) (string, []string, error)
	// GetJob 获取任务信息
	GetJob(requestId string, chainInfoId int, params []byte) ([]byte, error)
	// GetJobApprove 获取任务审批结果
	GetJobApprove(requestId string, chainInfoId int, params []byte) ([]byte, error)
	// DecryptPrivateData 解密密文数据
	DecryptPrivateData(localPlatformId string, data interface{}, common *types.Common) error
	// SetResultServerAddr 设置结果服务器地址
	SetResultServerAddr(requestId string, chainInfoId int32, partyId uint32, address string) error
	// GetPrivatePlatformPK 根据平台ID获取平台公钥
	GetPrivatePlatformPK(requestId string, chainInfoId, platformId int) (string, error)
	// InvokeContract 调用合约
	InvokeContract(requestId string, chainInfoId int, ContractName, method string, kvPairs []*chainpb.KeyValuePair) (string, error)
	// QueryContract 查询合约
	QueryContract(requestId string, chainInfoId int, ContractName, method string, kvPairs []*chainpb.KeyValuePair) (string, error)
}

var MiraIdaAccessServiceImpl MiraIdaAccessService

type miraIdaAccessService struct {
	cli pb.MiraIdaAccessClient
}

// InitMiraIdaAccessService 初始化MiraIdaAccessServiceImpl
func InitMiraIdaAccessService() error {
	caCrtFile := config.Conf.Mias.CaPath
	crtFile := config.Conf.Mias.CrtPath
	keyFile := config.Conf.Mias.KeyPath
	serverDNS := config.Conf.Mias.ServerDNS
	addr := config.Conf.Mias.Addr

	var (
		cli *zrpc.Client
		err error
	)
	for {
		// 创建GRPC客户端
		cli, err = grpc.CreateGRPCClient(caCrtFile, crtFile, keyFile, serverDNS, addr)
		if err == nil {
			break
		}
		logger.Warnf("create grpc client failed, retry later, err: %v", err)
		time.Sleep(3 * time.Second)
	}

	client := pb.NewMiraIdaAccessClient((*cli).Conn())
	mias := &miraIdaAccessService{
		cli: client,
	}
	MiraIdaAccessServiceImpl = mias
	return nil
}

// GetAssetList 获取资产列表
func (m miraIdaAccessService) GetAssetList(requestId string, chainInfoId int, req comm.ListInterfaceReq) (*comm.Pagination, []types.AssetItem, error) {
	request := &registrationpb.GetPrivateAssetListReq{
		RequestId:   requestId,
		PageNumber:  int32(req.PageNumber),
		PageSize:    int32(req.PageSize),
		Filters:     utils.Filters2PbFilters(req.Filters),
		ChainInfoId: int32(chainInfoId),
	}
	privateAssetListResp, err := m.cli.GetPrivateAssetList(context.Background(), request)
	if err != nil {
		logger.Errorf("get asset list failed, err: %v", err)
		return nil, nil, err
	}

	if privateAssetListResp.Code != http.StatusOK {
		return nil, nil, errors.Wrapf(errors.New(privateAssetListResp.Msg), "get asset list failed")
	}

	platformInfo, err := m.GetPlatformInfo(uint32(chainInfoId), requestId)
	if err != nil {
		logger.Errorf("get platformInfo failed, err: %v", err)
		return nil, nil, err
	}
	assetList := make([]types.AssetItem, 0)
	for _, asset := range privateAssetListResp.Data.List {
		assetList = append(assetList, types.AssetItem{
			AssetId:       fmt.Sprintf("%d", asset.AssetId),
			AssetNumber:   asset.AssetNumber,
			AssetName:     asset.AssetName,
			HolderCompany: asset.HolderCompany,
			Intro:         asset.Intro,
			TxId:          asset.TxId,
			UploadedAt:    asset.UploadedAt,
			ChainName:     platformInfo.ChainName,
		})
	}

	return &comm.Pagination{
		Total:      int(privateAssetListResp.Data.Pagination.Total),
		PageSize:   int(privateAssetListResp.Data.Pagination.PageSize),
		PageNumber: int(privateAssetListResp.Data.Pagination.PageNumber),
	}, assetList, nil
}

// GetAssetInfo 获取指定资产的信息
func (m miraIdaAccessService) GetAssetInfo(requestId string, chainInfoId int, assetId string) (*types.AssetInfo, error) {
	assetIdInt, err := strconv.Atoi(assetId)
	if err != nil {
		logger.Errorf("convert asset id to int failed, err: %v", err)
		return nil, err
	}

	request := &registrationpb.GetPrivateAssetInfoReq{
		RequestId:   requestId,
		AssetId:     int32(assetIdInt),
		ChainInfoId: int32(chainInfoId),
	}
	privateAssetInfoResp, err := m.cli.GetPrivateAssetInfo(context.Background(), request)
	if err != nil {
		logger.Errorf("get asset info failed, err: %v", err)
		return nil, err
	}

	if privateAssetInfoResp.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(privateAssetInfoResp.Msg), "get asset info failed")
	}

	baseInfo := privateAssetInfoResp.Data.ProductInfo.ProductBaseInfo
	dataInfo := privateAssetInfoResp.Data.ProductInfo.ProductDataInfo
	dataSet := privateAssetInfoResp.Data.ProductInfo.ProductDataSet
	assetInfo := &types.AssetInfo{
		AssetId:       string(privateAssetInfoResp.Data.AssetId),
		AssetNumber:   baseInfo.AssetNumber,
		AssetName:     baseInfo.AssetName,
		AssetEnName:   baseInfo.AssetEnName,
		AssetType:     int(baseInfo.AssetType),
		Scale:         fmt.Sprintf("%d%s", dataInfo.Scale, utils.ScaleTypeMap[int(dataInfo.ScaleType)]),
		Cycle:         fmt.Sprintf("%d%s", dataInfo.Cycle, utils.CycleUnitMap[int(dataInfo.Unit)]),
		TimeSpan:      dataInfo.TimeSpan,
		HolderCompany: privateAssetInfoResp.Data.HolderCompany,
		Intro:         baseInfo.Intro,
		//TxId:          privateAssetInfoResp.Data.ChainInfo.TxId,
		//UploadedAt:    privateAssetInfoResp.Data.ChainInfo.UploadedAt,
		DataInfo: types.DataInfo{
			DbName:    dataSet.DbConnInfo.DbName,
			TableName: dataSet.TableName,
		},
	}

	itemList := make([]types.SaveTableColumnItem, 0)
	for _, item := range dataSet.Items {
		itemList = append(itemList, types.SaveTableColumnItem{
			Name:         item.Name,
			DataType:     item.DataType,
			DataLength:   int(item.DataLength),
			Description:  item.Description,
			IsPrimaryKey: int(item.IsPrimaryKey),
			PrivacyQuery: int(item.PrivacyQuery),
		})
	}

	assetInfo.DataInfo.ItemList = itemList
	return assetInfo, nil
}

// GetAssetByEnName 通过资产英文名获取资产信息
func (m miraIdaAccessService) GetAssetByEnName(requestId string, chainInfoId int, assetEnName string) (*types.AssetInfo, error) {
	request := &registrationpb.GetPrivateAssetInfoByEnNameReq{
		RequestId:   requestId,
		AssetEnName: assetEnName,
		ChainInfoId: int32(chainInfoId),
	}
	privateAssetInfoResp, err := m.cli.GetPrivateAssetInfoByEnName(context.Background(), request)
	if err != nil {
		logger.Errorf("get asset info failed, err: %v", err)
		return nil, err
	}

	if privateAssetInfoResp.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(privateAssetInfoResp.Msg), "get asset info failed")
	}

	baseInfo := privateAssetInfoResp.Data.ProductInfo.ProductBaseInfo
	dataInfo := privateAssetInfoResp.Data.ProductInfo.ProductDataInfo
	dataSet := privateAssetInfoResp.Data.ProductInfo.ProductDataSet
	assetInfo := &types.AssetInfo{
		AssetId:       string(privateAssetInfoResp.Data.AssetId),
		AssetNumber:   baseInfo.AssetNumber,
		AssetName:     baseInfo.AssetName,
		AssetEnName:   baseInfo.AssetEnName,
		AssetType:     int(baseInfo.AssetType),
		Scale:         fmt.Sprintf("%d%s", dataInfo.Scale, utils.ScaleTypeMap[int(dataInfo.ScaleType)]),
		Cycle:         fmt.Sprintf("%d%s", dataInfo.Cycle, utils.CycleUnitMap[int(dataInfo.Unit)]),
		TimeSpan:      dataInfo.TimeSpan,
		HolderCompany: privateAssetInfoResp.Data.HolderCompany,
		Intro:         baseInfo.Intro,
		//TxId:          privateAssetInfoResp.Data.ChainInfo.TxId,
		//UploadedAt:    privateAssetInfoResp.Data.ChainInfo.UploadedAt,
		DataInfo: types.DataInfo{
			DbName:    dataSet.DbConnInfo.DbName,
			TableName: dataSet.TableName,
		},
	}

	itemList := make([]types.SaveTableColumnItem, 0)
	for _, item := range dataSet.Items {
		itemList = append(itemList, types.SaveTableColumnItem{
			Name:         item.Name,
			DataType:     item.DataType,
			DataLength:   int(item.DataLength),
			Description:  item.Description,
			IsPrimaryKey: int(item.IsPrimaryKey),
			PrivacyQuery: int(item.PrivacyQuery),
		})
	}

	assetInfo.DataInfo.ItemList = itemList
	return assetInfo, nil
}

// GetEnterpriseList 获取企业列表
func (m miraIdaAccessService) GetEnterpriseList(requestId string, chainInfoId int, assetName string) (*types.AssetEnterpriseList, error) {
	request := &registrationpb.GetPrivateEnterpriseAssetReq{
		AssetName:   assetName,
		ChainInfoId: int32(chainInfoId),
	}
	getPrivateEnterpriseAssetResp, err := m.cli.GetPrivateEnterpriseAsset(context.Background(), request)
	if err != nil {
		logger.Errorf("get asset info failed, err: %v", err)
		return nil, err
	}

	if getPrivateEnterpriseAssetResp.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(getPrivateEnterpriseAssetResp.Msg), "get asset info failed")
	}
	assetEnterpriseList := &types.AssetEnterpriseList{
		EnterpriseList: make([]*types.Enterprise, 0),
	}
	for _, privateEnterprise := range getPrivateEnterpriseAssetResp.Data {
		enterprise := &types.Enterprise{
			EnterpriseName:      privateEnterprise.EnterpriseName,
			EnterpriseAssetList: make([]*types.EnterpriseAsset, 0),
		}
		for _, privateAsset := range privateEnterprise.AssetList {
			enterpriseAsset := &types.EnterpriseAsset{
				AssetId:     strconv.Itoa(int(privateAsset.AssetId)),
				AssetNumber: privateAsset.AssetNumber,
				AssetName:   privateAsset.AssetName,
				AssetEnName: privateAsset.AssetEnName,
			}
			enterprise.EnterpriseAssetList = append(enterprise.EnterpriseAssetList, enterpriseAsset)
		}
		assetEnterpriseList.EnterpriseList = append(assetEnterpriseList.EnterpriseList, enterprise)
	}
	return assetEnterpriseList, nil
}

// CreateJob 任务上链
func (m miraIdaAccessService) CreateJob(requestId string, chainInfoId int, params []byte) (string, error) {
	kvPairs := []*chainpb.KeyValuePair{{
		Key:   comm.ParamJob,
		Value: params,
	}}
	request := &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: int32(chainInfoId),
		KvPairs:     kvPairs,
	}
	contractResponse, err := m.cli.CreateJob(context.Background(), request)
	if err != nil {
		logger.Errorf("create job failed, err: %v", err)
		return "", err
	}

	if contractResponse.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(contractResponse.Msg), "create job failed")
	}
	txId := contractResponse.GetChainDataInfo().GetTxId()

	return txId, nil
}

// CreateJobApprove 任务审批上链
func (m miraIdaAccessService) CreateJobApprove(requestId string, chainInfoId int, params []byte) (string, error) {
	kvPairs := []*chainpb.KeyValuePair{{
		Key:   comm.ParamJobApproveExpandInfo,
		Value: params,
	}}
	request := &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: int32(chainInfoId),
		KvPairs:     kvPairs,
	}
	contractResponse, err := m.cli.CreateJobApprove(context.Background(), request)
	if err != nil {
		logger.Errorf("create job approve failed, err: %v", err)
		return "", err
	}

	if contractResponse.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(contractResponse.Msg), "create job approve failed")
	}
	txId := contractResponse.GetChainDataInfo().GetTxId()

	return txId, err
}

// TriggerJobEnable 触发任务
func (m miraIdaAccessService) TriggerJobEnable(requestId string, chainInfoId int, jobId string) (string, error) {
	boolByte := make([]byte, 0)
	buf := bytes.NewBuffer(boolByte)
	binary.Write(buf, binary.LittleEndian, true)
	kvPairs := []*chainpb.KeyValuePair{
		{
			Key:   comm.ParamJobId,
			Value: []byte(jobId),
		},
		{
			Key:   comm.ParamTriggerEnable,
			Value: buf.Bytes(),
		},
	}
	request := &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: int32(chainInfoId),
		KvPairs:     kvPairs,
	}
	contractResponse, err := m.cli.UpdateJobTriggerEnable(context.Background(), request)
	if err != nil {
		logger.Errorf("create job failed, err: %v", err)
		return "", err
	}

	if contractResponse.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(contractResponse.Msg), "create job failed")
	}
	txId := contractResponse.GetChainDataInfo().GetTxId()

	return txId, nil
}

// CancelJobInstance 取消任务实例
func (m miraIdaAccessService) CancelJobInstance(requestId string, chainInfoId int, jobInstanceId string) (string, error) {
	boolByte := make([]byte, 0)
	buf := bytes.NewBuffer(boolByte)
	binary.Write(buf, binary.LittleEndian, true)
	kvPairs := []*chainpb.KeyValuePair{
		{
			Key:   comm.ParamJobInstanceId,
			Value: []byte(jobInstanceId),
		},
	}
	request := &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: int32(chainInfoId),
		KvPairs:     kvPairs,
	}
	contractResponse, err := m.cli.CancelJobInstance(context.Background(), request)
	if err != nil {
		logger.Errorf("cancel job instance failed, err: %v", err)
		return "", err
	}

	if contractResponse.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(contractResponse.Msg), "create job failed")
	}
	txId := contractResponse.GetChainDataInfo().GetTxId()

	return txId, nil
}

// GetPlatformInfo 获取平台信息
func (m miraIdaAccessService) GetPlatformInfo(chainInfoId uint32, requestId string) (*chainpb.PlatformData, error) {
	request := &chainpb.GetPlatformInfoRequest{
		ChainInfoId: chainInfoId,
		RequestId:   requestId,
	}
	resp, err := m.cli.GetPlatformInfo(context.Background(), request)
	if err != nil {
		logger.Errorf("get platform info failed, err: %v", err)
		return nil, err
	}

	if resp.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(resp.Msg), "get platform info failed")
	}

	return resp.GetData(), nil
}

// GetListChainInfo 获取平台信息
func (m miraIdaAccessService) GetListChainInfo(requestId string) (*chainpb.ListChainInfoData, error) {
	request := &chainpb.ListChainInfoRequest{
		RequestId: requestId,
	}
	resp, err := m.cli.ListChainInfo(context.Background(), request)
	if err != nil {
		logger.Errorf("get chain list info failed, err: %v", err)
		return nil, err
	}

	if resp.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(resp.Msg), "get chain list info failed")
	}

	return resp.GetData(), nil
}

// EncWithDeK 数字信封加密 使用公钥和额外参数加密明文，并返回密文和密钥
func (m miraIdaAccessService) EncWithDeK(plainText, pubKey, extraParam string, dataEnvelopAlgoType keypb.DataEnvelopAlgoType) (string, string, error) {
	request := &keypb.DataEnvelopeEncryptRequest{
		PlainText:           plainText,
		DataEnvelopAlgoType: dataEnvelopAlgoType,
		PubKey:              pubKey,
		ExtraParam:          extraParam,
	}
	resp, err := m.cli.EncWithDeK(context.Background(), request)
	if err != nil {
		logger.Errorf("enc with dek failed, err: %v", err)
		return "", "", err
	}

	if resp.Code != http.StatusOK {
		return "", "", errors.Wrapf(errors.New(resp.Msg), "enc with dek failed")
	}

	return resp.GetCipherText(), resp.GetKek(), nil
}

// DecByKeK 数字信封解密 使用密钥和公钥解密密文，返回明文
func (m miraIdaAccessService) DecByKeK(cipherText, pubKey, kek string, dataEnvelopAlgoType keypb.DataEnvelopAlgoType) (string, error) {
	request := &keypb.DataEnvelopeDecryptRequest{
		CipherText:          cipherText,
		DataEnvelopAlgoType: dataEnvelopAlgoType,
		PubKey:              pubKey,
		KeyId:               "",
		Kek:                 kek,
	}
	resp, err := m.cli.DecByKeK(context.Background(), request)
	if err != nil {
		logger.Errorf("dec by kek failed, err: %v", err)
		return "", err
	}

	if resp.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(resp.Msg), "dec by kek failed")
	}

	return resp.GetPlainText(), nil
}

// Encrypt 使用公钥和额外参数加密明文，返回密文
func (m miraIdaAccessService) Encrypt(plainText, pubKey, extraParam string, algoType keypb.AlgoType) (string, error) {
	request := &keypb.KeyEncryptRequest{
		PlainText:  plainText,
		AlgoType:   algoType,
		PubKey:     pubKey,
		ExtraParam: extraParam,
	}
	resp, err := m.cli.Encrypt(context.Background(), request)
	if err != nil {
		logger.Errorf("enc with dek failed, err: %v", err)
		return "", err
	}

	if resp.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(resp.Msg), "enc with dek failed")
	}

	return resp.GetCipherText(), nil
}

// Decrypt 使用公钥解密密文，返回明文。
func (m miraIdaAccessService) Decrypt(cipherText, pubKey string, algoType keypb.AlgoType) (string, error) {
	request := &keypb.KeyDecryptRequest{
		CipherText: cipherText,
		AlgoType:   algoType,
		PubKey:     pubKey,
	}
	resp, err := m.cli.Decrypt(context.Background(), request)
	if err != nil {
		logger.Errorf("enc with dek failed, err: %v", err)
		return "", err
	}

	if resp.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(resp.Msg), "enc with dek failed")
	}
	return resp.GetPlainText(), nil
}

// EncWithDeKWithPkList 数字信封传入多个公钥，对同一个明文加密
func (m miraIdaAccessService) EncWithDeKWithPkList(plainText, extraParam string, pubKeyList []string, dataEnvelopAlgoType keypb.DataEnvelopAlgoType) (string, []string, error) {
	request := &keypb.DataEnvelopeEncryptWithPkListRequest{
		PlainText:           plainText,
		DataEnvelopAlgoType: dataEnvelopAlgoType,
		PubKeyList:          pubKeyList,
		ExtraParam:          extraParam,
	}
	resp, err := m.cli.EncWithDeKWithPkList(context.Background(), request)
	if err != nil {
		logger.Errorf("enc with dek with pk list failed, err: %v", err)
		return "", nil, err
	}

	if resp.Code != http.StatusOK {
		return "", nil, errors.Wrapf(errors.New(resp.Msg), "enc with dek with pk list failed")
	}

	return resp.GetCipherText(), resp.GetKekList(), nil
}

// GetJob 获取任务信息
func (m miraIdaAccessService) GetJob(requestId string, chainInfoId int, params []byte) ([]byte, error) {
	kvPairs := []*chainpb.KeyValuePair{{
		Key:   comm.ParamJobId,
		Value: params,
	}}
	request := &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: int32(chainInfoId),
		KvPairs:     kvPairs,
	}
	contractResponse, err := m.cli.GetJob(context.Background(), request)
	if err != nil {
		logger.Errorf("get job failed, err: %v", err)
		return nil, err
	}

	if contractResponse.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(contractResponse.Msg), "get job failed")
	}
	result := contractResponse.GetChainDataInfo().GetResult()

	return result, err
}

// GetJobApprove 获取任务审批信息
func (m miraIdaAccessService) GetJobApprove(requestId string, chainInfoId int, params []byte) ([]byte, error) {
	kvPairs := []*chainpb.KeyValuePair{{
		Key:   comm.ParamJobId,
		Value: params,
	}}
	request := &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: int32(chainInfoId),
		KvPairs:     kvPairs,
	}
	contractResponse, err := m.cli.GetJobApprove(context.Background(), request)
	if err != nil {
		logger.Errorf("get job approve failed, err: %v", err)
		return nil, err
	}

	if contractResponse.Code != http.StatusOK {
		return nil, errors.Wrapf(errors.New(contractResponse.Msg), "get job approve failed")
	}
	result := contractResponse.GetChainDataInfo().GetResult()

	return result, err
}

// DecryptPrivateData 解密密文数据
func (m miraIdaAccessService) DecryptPrivateData(localPlatformId string, data interface{}, common *types.Common) error {
	// 公共信息为空
	if common == nil {
		return errors.New("invalid enc data")
	}
	// check params
	if len(common.PartyIdList) != len(common.PkList) || len(common.PartyIdList) != len(common.KekList) {
		return errors.New("invalid partyIdList, pkList or kekList, length not equal")
	}

	// 需要解密
	if common.IsEncrypted {
		var myKek, myPk string
		for i, partyId := range common.PartyIdList {
			if partyId == localPlatformId {
				myKek = common.KekList[i]
				myPk = common.PkList[i]
				break
			}
		}
		if len(myKek) == 0 {
			return errors.New("my kek not found")
		}
		if len(myPk) == 0 {
			return errors.New("my pk not found")
		}

		// 解密
		plainText, err := m.DecByKeK(common.CipherData, myPk, myKek, keypb.DataEnvelopAlgoType(keypb.DataEnvelopAlgoType_value[common.Algo]))
		if err != nil {
			return err
		}
		err = json.Unmarshal([]byte(plainText), &data)
		if err != nil {
			return err
		}
	}

	return nil
}

func (m miraIdaAccessService) SetResultServerAddr(requestId string, chainInfoId int32, partyId uint32, address string) error {
	resp, err := m.cli.SetResultServerAddr(context.Background(), &pb.ContractRequest{
		RequestId:   requestId,
		ChainInfoId: chainInfoId,
		KvPairs: []*chainpb.KeyValuePair{
			{
				Key:   comm.ParamPartyId,
				Value: []byte(strconv.Itoa(int(partyId))),
			},
			{
				Key:   comm.ParamServerAddr,
				Value: []byte(address),
			},
		},
	})
	if err != nil {
		logger.Errorf("set result server addr failed, err: %v", err)
		return err
	}
	if resp.Code != http.StatusOK {
		return errors.Wrapf(errors.New(resp.Msg), "set result server addr failed")
	}
	return nil
}

// GetPrivatePlatformPK 根据平台ID获取平台公钥
func (m miraIdaAccessService) GetPrivatePlatformPK(requestId string, chainInfoId, platformId int) (string, error) {

	request := &registrationpb.GetPrivatePlatformPKReq{
		RequestId:   requestId,
		PlatformId:  int32(platformId),
		ChainInfoId: int32(chainInfoId),
	}
	getPrivatePlatformPKResponse, err := m.cli.GetPrivatePlatformPK(context.Background(), request)
	if err != nil {
		logger.Errorf("get job approve failed, err: %v", err)
		return "", err
	}

	if getPrivatePlatformPKResponse.Code != http.StatusOK {
		return "", errors.Wrapf(errors.New(getPrivatePlatformPKResponse.Msg), "get private platform pk failed")
	}
	if err != nil {
		logger.Errorf("get private platform pk failed, err: %v", err)
		return "", err
	}

	publicKey := getPrivatePlatformPKResponse.Data.PublicKey

	return publicKey, err
}
